/*
 * Toolkit GUI, an application built for operating pinkRF's signal generators.
 *
 * Contact: https://www.pinkrf.com/contact/
 * Copyright © 2018-2024 pinkRF B.V
 * GNU General Public License version 3.
 *
 * This program is free software: you can redistribute it and/or modify it under the terms of the GNU General Public License as published by the Free Software Foundation, either version 3 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with this program. If not, see https://www.gnu.org/licenses/
 *
 * Author: Iordan Svechtarov
 *
 */

#include "gui_1channel.h"
#include "ui_gui_1channel.h"
#include "miscellaneous.h"
#include "serial_v2.h"
#include <iostream>
#include <math.h>
#include <QCoreApplication>
#include <QCursor>
#include <QDebug>
#include <QDir>
#include <QMessageBox>
#include <QNetworkInterface>
#include <QTimer>
#include <QThread>

QT_USE_NAMESPACE

GUI_1channel::GUI_1channel(QWidget *parent) : QMainWindow(parent), ui(new Ui::GUI_1channel)
{
	ui->setupUi(this);

	/* Set the Toolkit GUI version */
	toolkit_GUI_version = "2.7.9.1";
	ui->GUI_version_label_2->setText(toolkit_GUI_version);
	qDebug() << "Toolkit GUI " << toolkit_GUI_version;

	/* Get the names of the serialports, as well as all the other startup values from config.txt */
	config = new ConfigHandler(QCoreApplication::applicationDirPath() + "/config.txt");

	/* Display versioning info */
	if(config->get_support_splitter_controls_attenuation() == true)	//Splitter / NChannel attenuation feature require firmware version 2.8.14 or higher
	{
		firmware_version_requirement[0] = 2;
		firmware_version_requirement[1] = 8;
		firmware_version_requirement[2] = 14;
		firmware_version_requirement[3] = 0;
	}
	else if(config->get_splitter_channel_count() >= 2)	//Splitter / NChannel features require firmware version 2.8.10 or higher
	{
		firmware_version_requirement[0] = 2;
		firmware_version_requirement[1] = 8;
		firmware_version_requirement[2] = 10;
		firmware_version_requirement[3] = 0;
	}
	else if(config->get_support_power_offset() == true)	//Power offset Feature requires firmware version 2.8.0
	{
		firmware_version_requirement[0] = 2;
		firmware_version_requirement[1] = 8;
		firmware_version_requirement[2] = 0;
		firmware_version_requirement[3] = 0;
	}

	else if(config->get_support_external_triggering_mode() == true)		//(New) External Triggering implementation requires firmware version 2.7.0
	{
		firmware_version_requirement[0] = 2;
		firmware_version_requirement[1] = 7;
		firmware_version_requirement[2] = 0;
		firmware_version_requirement[3] = 0;
	}
	else
	{
		firmware_version_requirement[0] = 1;
		firmware_version_requirement[1] = 6;
		firmware_version_requirement[2] = 4;
		firmware_version_requirement[3] = 0;
	}

	/* Set up Data Logging */
	QStringList loggercolumns = {"RF_enable",
								 "Frequency",
								 "Power_Setpoint_watt",
								 "Power_FWD_watt",
								 "Power_RFL_watt",
								 "S11_Reflection",
								 "PWM_Frequency",
								 "PWM_Duty_Cycle",
								 "PA_Temperature",
								 "SGx_Error",
								 "GCG",
								 "MCG",
								 "SG_Temperature",
								 "Voltage_measurement",
								 "Current_measurement"
								};

	for (int i = 1; i < (config->get_PSU_count() + 1); i++)
	{
//		loggercolumns.append("PSU_Enable_" + QString::number(i));
		loggercolumns.append("PSU_Voltage_" + QString::number(i));
		loggercolumns.append("PSU_Current_" + QString::number(i));
		loggercolumns.append("PSU_Power_" + QString::number(i));
	}
	loggercolumns.append("PSU_Power_Efficiency");
	loggercolumns.append("PSU_Power_Dissipation");

	datalogger = new LoggingClass(config->get_logging_rate(), config->get_logging_directory(), "log", loggercolumns);


	/*******************************************************************
	 * Setup channels
	 *******************************************************************/
	RF_system_constructor = new RF_System_1channel();

	//Connect modbus enable
	connect(this, &GUI_1channel::signal_set_modbus_enable, RF_system_constructor->mbServer, &ModbusServer::set_listen_enable);
	connect(RF_system_constructor->mbServer, &ModbusServer::signal_modbus_connected_enable, this, &GUI_1channel::handler_modbus_connected_enable);

	//Connect EtherCAT enable
#ifdef ETHERCAT
	connect(this, &GUI_1channel::signal_set_ethercat_enable, &RF_system_constructor->ethercatModule, &EtherCATModule::setEnabled);
#endif

	/* Activate Channels */
	for (int i = 0; i < RF_System::Channels->count(); i++)
	{
		connect(this, &GUI_1channel::signal_channelInit, RF_System::Channels->at(i), &RF_Channel::Initialize);
		connect(this, &GUI_1channel::signal_timerStart, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Start);
		connect(this, &GUI_1channel::signal_timerStop, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Stop);
		connect(this, &GUI_1channel::signal_execute_reset_SGx, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_SGx);
		connect(this, &GUI_1channel::signal_execute_error_clear, RF_System::Channels->at(i), &RF_Channel::Execute_Error_Clear);
		connect(this, &GUI_1channel::signal_set_error_clearing_enable, RF_System::Channels->at(i), &RF_Channel::Set_Error_Clearing_Enable);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_firmware_version_get, this, &GUI_1channel::handler_firmware_version_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_board_identity_get, this, &GUI_1channel::handler_board_identity_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_hardware_identity_get, this, &GUI_1channel::handler_hardware_identity_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_type_get, this, &GUI_1channel::handler_PA_type_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_error_get, this, &GUI_1channel::handler_error_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_reset_detected, this, &GUI_1channel::handler_reset_detected);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_reset_detected_runtime, this, &GUI_1channel::handler_reset_detected);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_PA_get, this, &GUI_1channel::handler_temperature_PA_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_temperature_SG_get, this, &GUI_1channel::handler_temperature_SG_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_control_mode_get, this, &GUI_1channel::handler_power_control_mode_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_clock_source_get, this, &GUI_1channel::handler_clock_source_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_RF_enable_get, this, &GUI_1channel::handler_RF_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings, this, &GUI_1channel::handler_PA_power_readings);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PA_power_readings_NChannel, this, &GUI_1channel::handler_PA_power_readings_NChannel);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_power_get, this, &GUI_1channel::handler_power_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_VGA_attenuation_get, this, &GUI_1channel::handler_VGA_attenuation_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_IQMod_magnitude_get, this, &GUI_1channel::handler_IQMod_magnitude_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_frequency_get, this, &GUI_1channel::handler_frequency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_offset_db_get, this, &GUI_1channel::handler_offset_db_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_phase_get, this, &GUI_1channel::handler_phase_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_splitter_configuration, this, &GUI_1channel::handler_splitter_configuration);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_CW_enable_get, this, &GUI_1channel::handler_CW_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DLL_enable_get, this, &GUI_1channel::handler_DLL_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DLL_settings_get, this, &GUI_1channel::handler_DLL_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_ALL_compatible_get, this, &GUI_1channel::handler_ALL_compatible_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_ALL_enable_get, this, &GUI_1channel::handler_ALL_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_ALL_settings_get, this, &GUI_1channel::handler_ALL_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DVGA_compatible_get, this, &GUI_1channel::handler_DVGA_compatible_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DVGA_forward_settings_get, this, &GUI_1channel::handler_DVGA_forward_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_DVGA_reflected_settings_get, this, &GUI_1channel::handler_DVGA_reflected_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PWM_settings_get, this, &GUI_1channel::handler_PWM_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_external_triggering_enable_get, this, &GUI_1channel::handler_external_triggering_enable_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_settings_get, this, &GUI_1channel::handler_SWP_settings_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_SWP_measurements_get, this, &GUI_1channel::handler_SWP_measurement_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_voltage_get, this, &GUI_1channel::handler_PSU_voltage_measurement_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_current_get, this, &GUI_1channel::handler_PSU_current_measurement_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_IU_reading_get, this, &GUI_1channel::handler_PSU_IU_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_power_efficiency_get, this, &GUI_1channel::handler_PSU_power_efficiency_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_enable_combined_get, this, &GUI_1channel::handler_PSU_enable_combined_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_voltage_setpoint_get, this, &GUI_1channel::handler_PSU_voltage_setpoint_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_current_limit_get, this, &GUI_1channel::handler_PSU_current_limit_get);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_PSU_dissipation_get, this, &GUI_1channel::handler_PSU_dissipation_get);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_amplifier_mode, this, &GUI_1channel::handler_availability_amplifier_mode);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_external_triggering, this, &GUI_1channel::handler_availability_external_triggering);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_PWM, this, &GUI_1channel::handler_availability_PWM);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_DLL, this, &GUI_1channel::handler_availability_DLL);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_ALL, this, &GUI_1channel::handler_availability_ALL);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_sweep, this, &GUI_1channel::handler_availability_sweep);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_availability_PSU_enable, this, &GUI_1channel::handler_availability_PSU_enable);

		connect(this, &GUI_1channel::signal_request_PA_power_update, RF_System::Channels->at(i), &RF_Channel::Update_PA_Power);

		connect(this, &GUI_1channel::signal_RF_enable, RF_System::Channels->at(i), &RF_Channel::Set_RF_enable);
		connect(this, &GUI_1channel::signal_setClock_source, RF_System::Channels->at(i), &RF_Channel::Set_Clock_Source);
		connect(this, &GUI_1channel::signal_set_power_control_mode, RF_System::Channels->at(i), &RF_Channel::Set_Power_Control_Mode);
		connect(this, &GUI_1channel::signal_setPWM_triggering_mode, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Triggering);
		connect(this, &GUI_1channel::signal_setExternal_triggering_enable, RF_System::Channels->at(i), &RF_Channel::Set_External_Triggering_Enable_Configuration);

		connect(this, &GUI_1channel::signal_setVGA_attenuation, RF_System::Channels->at(i), &RF_Channel::Set_VGA_attenuation);
		connect(this, &GUI_1channel::signal_setIQMod_magnitude, RF_System::Channels->at(i), &RF_Channel::Set_IQMod_Magnitude);
		connect(this, &GUI_1channel::signal_setPowerWatt, RF_System::Channels->at(i), &RF_Channel::Set_Power_Watt);
		connect(this, &GUI_1channel::signal_setPowerdBm, RF_System::Channels->at(i), &RF_Channel::Set_Power_dBm);
		connect(this, &GUI_1channel::signal_setFrequency, RF_System::Channels->at(i), &RF_Channel::Set_Frequency);
		connect(this, &GUI_1channel::signal_setPhase, RF_System::Channels->at(i), &RF_Channel::Set_Phase);
		connect(this, &GUI_1channel::signal_setOffsetdB, RF_System::Channels->at(i), &RF_Channel::Set_Offset_dB);

//		connect(this, &GUI_1channel::signal_setCW_enable, RF_System::Channels->at(i), &RF_Channel::Set_CW_enable);
		connect(this, &GUI_1channel::signal_setPWM_enable, RF_System::Channels->at(i), &RF_Channel::Set_PWM_enable);
		connect(this, &GUI_1channel::signal_setPWM_frequency, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Frequency);
		connect(this, &GUI_1channel::signal_setPWM_duty_cycle, RF_System::Channels->at(i), &RF_Channel::Set_PWM_Duty_Cycle);

		connect(this, &GUI_1channel::signal_setDLL_enable, RF_System::Channels->at(i), &RF_Channel::Set_DLL_enable);
		connect(this, &GUI_1channel::signal_setDLL_frequency_limit_lower, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Lower_Limit);
		connect(this, &GUI_1channel::signal_setDLL_frequency_limit_upper, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Upper_Limit);
		connect(this, &GUI_1channel::signal_setDLL_start_frequency, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Start);
		connect(this, &GUI_1channel::signal_setDLL_step_frequency, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Step);
		connect(this, &GUI_1channel::signal_setDLL_threshold, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Threshold);
		connect(this, &GUI_1channel::signal_setDLL_main_delay, RF_System::Channels->at(i), &RF_Channel::Set_DLL_Frequency_Main_Delay);

		connect(this, &GUI_1channel::signal_setALL_enable, RF_System::Channels->at(i), &RF_Channel::Set_ALL_enable);
		connect(this, &GUI_1channel::signal_setALL_frequency_limit_upper, RF_System::Channels->at(i), &RF_Channel::Set_ALL_Frequency_Upper_Limit);
		connect(this, &GUI_1channel::signal_setALL_frequency_limit_lower, RF_System::Channels->at(i), &RF_Channel::Set_ALL_Frequency_Lower_Limit);
		connect(this, &GUI_1channel::signal_setALL_threshold, RF_System::Channels->at(i), &RF_Channel::Set_ALL_Frequency_Threshold);

		connect(this, &GUI_1channel::signal_setDVGA_amplifier_forward_enable, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Amplifier_Forward_Enable);
		connect(this, &GUI_1channel::signal_setDVGA_amplifier_reflected_enable, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Amplifier_Reflected_Enable);
		connect(this, &GUI_1channel::signal_setDVGA_attentuation_forward, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Attenuation_Forward);
		connect(this, &GUI_1channel::signal_setDVGA_attentuation_reflected, RF_System::Channels->at(i), &RF_Channel::Set_DVGA_Attenuation_Reflected);

		connect(this, &GUI_1channel::signal_setSWP_power_dbm, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Power_dBm);
		connect(this, &GUI_1channel::signal_setSWP_power_watt, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Power_Watt);
		connect(this, &GUI_1channel::signal_setSWP_start_frequency, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Start);
		connect(this, &GUI_1channel::signal_setSWP_stop_frequency, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Stop);
		connect(this, &GUI_1channel::signal_setSWP_step_frequency, RF_System::Channels->at(i), &RF_Channel::Set_SWP_Frequency_Step);
		connect(this, &GUI_1channel::signal_execute_SWP, RF_System::Channels->at(i), &RF_Channel::Execute_SWP_dBm);

		connect(this, &GUI_1channel::signal_setPSU_enable, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Enable_Safely);
		connect(this, &GUI_1channel::signal_setPSU_voltage, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Voltage_Setpoint);
		connect(this, &GUI_1channel::signal_setPSU_current_limit, RF_System::Channels->at(i), &RF_Channel::Set_PSU_Current_Limit);

		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_working, this, &GUI_1channel::handler_channel_working);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_channel_initialization_complete, this, &GUI_1channel::handler_channel_initialization_complete);
		connect(RF_System::Channels->at(i), &RF_Channel::signal_notification, this, &GUI_1channel::handler_notification_get);

		connect(datalogger, &LoggingClass::signal_datalogging_enable, this, &GUI_1channel::handler_datalogging_enable_get);
		connect(datalogger, &LoggingClass::signal_datalogging_storage_sufficient, this, &GUI_1channel::handler_datalogging_storage_sufficient);
		connect(datalogger, &LoggingClass::signal_datalogging_log_deleted, this, &GUI_1channel::handler_datalogging_log_deleted);

		/* RCM Handling */
		connect(this, &GUI_1channel::signal_remote_command_mode_set, RF_system_constructor->Remote_Command_Mode, &RCM_Class::Handler_Remote_Command_Mode_Set);
		connect(RF_system_constructor->Remote_Command_Mode, &RCM_Class::signal_set_error_clearing_enable, this,  &GUI_1channel::handler_RCM_error_clearing);
		connect(RF_system_constructor->Remote_Command_Mode, &RCM_Class::signal_RCM_notification, this, &GUI_1channel::handler_RCM_notification);
		connect(RF_system_constructor->Remote_Command_Mode, &RCM_Class::signal_RCM_mode, this, &GUI_1channel::handler_remote_command_mode_get);
		connect(RF_system_constructor->Remote_Command_Mode, &RCM_Class::signal_execute_reset_SGx, RF_System::Channels->at(i), &RF_Channel::Execute_Reset_SGx);
		connect(RF_system_constructor->Remote_Command_Mode, &RCM_Class::signal_timerStart, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Start);
		connect(RF_system_constructor->Remote_Command_Mode, &RCM_Class::signal_timerStop, RF_System::Channels->at(i), &RF_Channel::Timer_Readings_Stop);
		
		/* Modbus interactions */
		connect(RF_system_constructor->mbServer, &ModbusServer::signal_set_datalogging_enable, this, &GUI_1channel::handler_datalogging);
		connect(datalogger, &LoggingClass::signal_datalogging_enable, RF_system_constructor->mbServer, &ModbusServer::handler_datalogging_enable_get);
		connect(RF_system_constructor->mbServer, &ModbusServer::signal_execute_restart_program, this, &GUI_1channel::handler_execute_restart_program);
		connect(datalogger, &LoggingClass::signal_datalogging_storage_sufficient, RF_system_constructor->mbServer, &ModbusServer::handler_datalogging_storage_sufficient);

		emit signal_channelInit(i+1);
		emit signal_timerStart(i+1, RF_system_constructor->config->get_polling_rate_data());
	}

	/* Populate the QFrame list for the NChannel splitter frames */
	frame_list.append(ui->frame_splitter_CH1);
	frame_list.append(ui->frame_splitter_CH2);
	frame_list.append(ui->frame_splitter_CH3);
	frame_list.append(ui->frame_splitter_CH4);
	frame_list.append(ui->frame_splitter_CH5);
	frame_list.append(ui->frame_splitter_CH6);
	frame_list.append(ui->frame_splitter_CH7);
	frame_list.append(ui->frame_splitter_CH8);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH1);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH2);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH3);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH4);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH5);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH6);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH7);
	splitter_button_magnitude_list.append(ui->pushButton_splitter_magnitude_CH8);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH1);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH2);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH3);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH4);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH5);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH6);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH7);
	splitter_button_attenuation_list.append(ui->pushButton_splitter_attenuation_CH8);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH1);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH2);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH3);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH4);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH5);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH6);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH7);
	splitter_button_phase_list.append(ui->pushButton_splitter_phase_CH8);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH1);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH2);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH3);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH4);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH5);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH6);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH7);
	splitter_label_magnitude_list.append(ui->label_splitter_magnitude_CH8);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH1);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH2);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH3);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH4);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH5);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH6);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH7);
	splitter_label_attenuation_list.append(ui->label_splitter_attenuation_CH8);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH1);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH2);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH3);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH4);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH5);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH6);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH7);
	splitter_label_phase_list.append(ui->label_splitter_phase_CH8);

	/* set up button groups, make it so only one button can be checked at a time */
	menu_buttonGroup.addButton(ui->menu_home_Button);
	menu_buttonGroup.addButton(ui->menu_splitter_Button);
	menu_buttonGroup.addButton(ui->menu_ALL_Button);
	menu_buttonGroup.addButton(ui->menu_DLL_Button);
	menu_buttonGroup.addButton(ui->menu_sweep_Button);
	menu_buttonGroup.addButton(ui->menu_PSU_Button);
	menu_buttonGroup.addButton(ui->menu_settings_Button);
	menu_buttonGroup.addButton(ui->menu_help_Button);
	menu_buttonGroup.addButton(ui->menu_power_Button);
	menu_buttonGroup.setExclusive(true);

	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_standalone_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_master_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_slave_end_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_slave_inline_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_AUX_master_1);
	clock_source_buttonGroup.addButton(ui->pushButton_clock_source_AUX_slave_end_1);
	clock_source_buttonGroup.setExclusive(true);

	RCM_buttonGroup.addButton(ui->pushButton_remote_command_OFF_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_USB_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_USB_blind_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_TCP_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_TCP_blind_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_RS232_1);
	RCM_buttonGroup.addButton(ui->pushButton_remote_command_RS232_blind_1);
	RCM_buttonGroup.setExclusive(true);

	modbus_buttonGroup.addButton(ui->pushButton_modbus_OFF_1);
	modbus_buttonGroup.addButton(ui->pushButton_modbus_TCP_slave_1);
	modbus_buttonGroup.setExclusive(true);

	ethercat_buttonGroup.addButton(ui->pushButton_etherCAT_OFF_1);
	ethercat_buttonGroup.addButton(ui->pushButton_etherCAT_ON_1);
	ethercat_buttonGroup.setExclusive(true);

	PWM_triggering_buttonGroup.addButton(ui->pushButton_PWM_triggering_free_running_1);
	PWM_triggering_buttonGroup.addButton(ui->pushButton_PWM_triggering_master_1);
	PWM_triggering_buttonGroup.addButton(ui->pushButton_PWM_triggering_slave_1);
	PWM_triggering_buttonGroup.setExclusive(true);

	power_control_buttonGroup.addButton(ui->pushButton_power_control_normal_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_analog_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_feedforward_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_amplifier_1);
	power_control_buttonGroup.addButton(ui->pushButton_power_control_PID_1);
	power_control_buttonGroup.setExclusive(true);

	External_triggering_buttonGroup.addButton(ui->pushButton_external_triggering_OFF_1);
	External_triggering_buttonGroup.addButton(ui->pushButton_external_triggering_ON_1);
	External_triggering_buttonGroup.setExclusive(true);

	logging_buttonGroup.addButton(ui->pushButton_data_logging_OFF_1);
	logging_buttonGroup.addButton(ui->pushButton_data_logging_ON_1);
	logging_buttonGroup.setExclusive(true);

	/* set maximum rows on the Settings menu plainTextEdit. */
	ui->settings_plainTextEdit->setMaximumBlockCount(100);

	/* set Watt default */
	set_power_reading_units_watt(true);	//true = watt, false = dBm

	/* set Home menu default */
	ui->menu_home_Button->click();		//Always start from the Home menu.

	/* hide the DLL frequency indicator by default */
	ui->label_DLL_frequency_lock_1->setVisible(false);
	ui->label_DLL_frequency_lock_2->setVisible(false);
	ui->label_DLL_frequency_lock_3->setVisible(false);

	/* Hide the notifications by default */
	ui->label_notification->setVisible(false);
	ui->label_error_message->setVisible(false);
	ui->label_low_storage_1->setVisible(false);

	//
	// TODO:
	// ODC implementation also uses the state of phase_gain_board_mode config item to disable phase stuff. See if that makes sense here
	//
	/* Phase control */
	ui->phase_label->setEnabled(config->get_support_phase_control());
	ui->phase_label->setVisible(config->get_support_phase_control());
	ui->pushButton_phase_1->setEnabled(config->get_support_phase_control());
	ui->pushButton_phase_1->setVisible(config->get_support_phase_control());


	/* Declare the timers. */
	notification_timer = new QTimer(this);		//Used for displaying temporary warning messages.

	/* Show or hide the various mode controls in settings menu */
	/* WARNING: PWM triggering does not work in firmware! */
	ui->pushButton_PWM_triggering_free_running_1->setVisible(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_free_running_1->setEnabled(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_master_1->setVisible(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_master_1->setEnabled(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_slave_1->setVisible(config->get_support_PWM_triggering());
	ui->pushButton_PWM_triggering_slave_1->setEnabled(config->get_support_PWM_triggering());
	ui->PWM_triggering_label->setVisible(config->get_support_PWM_triggering());
	ui->PWM_triggering_label->setEnabled(config->get_support_PWM_triggering());

	coherency_supported = (config->get_support_coherency_standalone() ||
						   config->get_support_coherency_LVDS_master() ||
						   config->get_support_coherency_LVDS_slave_end() ||
						   config->get_support_coherency_LVDS_slave_inline() ||
						   config->get_support_coherency_AUX_master() ||
						   config->get_support_coherency_AUX_slave_end());

	ui->pushButton_clock_source_standalone_1->setVisible(config->get_support_coherency_standalone());
	ui->pushButton_clock_source_standalone_1->setEnabled(config->get_support_coherency_standalone());
	ui->pushButton_clock_source_master_1->setVisible(config->get_support_coherency_LVDS_master());
	ui->pushButton_clock_source_master_1->setEnabled(config->get_support_coherency_LVDS_master());
	ui->pushButton_clock_source_slave_end_1->setVisible(config->get_support_coherency_LVDS_slave_end());
	ui->pushButton_clock_source_slave_end_1->setEnabled(config->get_support_coherency_LVDS_slave_end());
	ui->pushButton_clock_source_slave_inline_1->setVisible(config->get_support_coherency_LVDS_slave_inline());
	ui->pushButton_clock_source_slave_inline_1->setEnabled(config->get_support_coherency_LVDS_slave_inline());
	ui->pushButton_clock_source_AUX_master_1->setVisible(config->get_support_coherency_AUX_master());
	ui->pushButton_clock_source_AUX_master_1->setEnabled(config->get_support_coherency_AUX_master());
	ui->pushButton_clock_source_AUX_slave_end_1->setVisible(config->get_support_coherency_AUX_slave_end());
	ui->pushButton_clock_source_AUX_slave_end_1->setEnabled(config->get_support_coherency_AUX_slave_end());
	ui->label_clock_source_1->setEnabled(coherency_supported);
	ui->label_clock_source_1->setVisible(coherency_supported);

	RCM_supported = (config->get_support_RCM_mode_USB_live() ||
					 config->get_support_RCM_mode_USB_blind() ||
					 config->get_support_RCM_mode_RS232_live() ||
					 config->get_support_RCM_mode_RS232_blind() ||
					 config->get_support_RCM_mode_TCP_live() ||
					 config->get_support_RCM_mode_TCP_blind());

	ui->RCM_label->setEnabled(RCM_supported);
	ui->RCM_label->setVisible(RCM_supported);
	ui->pushButton_remote_command_OFF_1->setVisible(RCM_supported);
	ui->pushButton_remote_command_OFF_1->setEnabled(RCM_supported);
	ui->pushButton_remote_command_USB_1->setVisible(config->get_support_RCM_mode_USB_live());
	ui->pushButton_remote_command_USB_1->setEnabled(config->get_support_RCM_mode_USB_live());
	ui->pushButton_remote_command_USB_blind_1->setVisible(config->get_support_RCM_mode_USB_blind());
	ui->pushButton_remote_command_USB_blind_1->setEnabled(config->get_support_RCM_mode_USB_blind());
	ui->pushButton_remote_command_RS232_1->setVisible(config->get_support_RCM_mode_RS232_live());
	ui->pushButton_remote_command_RS232_1->setEnabled(config->get_support_RCM_mode_RS232_live());
	ui->pushButton_remote_command_RS232_blind_1->setVisible(config->get_support_RCM_mode_RS232_blind());
	ui->pushButton_remote_command_RS232_blind_1->setEnabled(config->get_support_RCM_mode_RS232_blind());
	ui->pushButton_remote_command_TCP_1->setVisible(config->get_support_RCM_mode_TCP_live());
	ui->pushButton_remote_command_TCP_1->setEnabled(config->get_support_RCM_mode_TCP_live());
	ui->pushButton_remote_command_TCP_blind_1->setVisible(config->get_support_RCM_mode_TCP_blind());
	ui->pushButton_remote_command_TCP_blind_1->setEnabled(config->get_support_RCM_mode_TCP_blind());
	//
	// TODO:
	// RS485 RCM is currently unsupported!
	//	-> Don't forget to add it as a factor for RCM_support!
	//


#ifndef DEBUG

#elif DEBUG
//	ui->pushButton_remote_command_RS485_1->setVisible(config->get_support_RCM_mode_RS485_live());
//	ui->pushButton_remote_command_RS485_1->setEnabled(config->get_support_RCM_mode_RS485_live());
//	ui->pushButton_remote_command_RS485_blind_1->setVisible(config->get_support_RCM_RS485_blind_mode());
//	ui->pushButton_remote_command_RS485_blind_1->setEnabled(config->get_support_RCM_RS485_blind_mode());
#endif


	ui->pushButton_modbus_OFF_1->setVisible(config->get_support_modbus_mode());
	ui->pushButton_modbus_OFF_1->setEnabled(config->get_support_modbus_mode());
	ui->pushButton_modbus_TCP_slave_1->setVisible(config->get_support_modbus_mode());
	ui->pushButton_modbus_TCP_slave_1->setEnabled(config->get_support_modbus_mode());
	ui->label_modbus_mode_1->setVisible(config->get_support_modbus_mode());
	ui->label_modbus_mode_1->setEnabled(config->get_support_modbus_mode());


#if defined ETHERCAT
	ui->pushButton_etherCAT_OFF_1->setVisible(config->get_support_ethercat_mode());
	ui->pushButton_etherCAT_ON_1->setVisible(config->get_support_ethercat_mode());
	ui->label_etherCAT_mode_1->setVisible(config->get_support_ethercat_mode());
#else
	ui->pushButton_etherCAT_OFF_1->setVisible(false);
	ui->pushButton_etherCAT_ON_1->setVisible(false);
	ui->label_etherCAT_mode_1->setVisible(false);
#endif


	ui->pushButton_power_control_normal_1->setVisible(config->get_support_power_control_modes());
	ui->pushButton_power_control_normal_1->setEnabled(config->get_support_power_control_modes());
	ui->pushButton_power_control_analog_1->setVisible(config->get_support_power_control_modes() && config->get_support_AIS_mode());
	ui->pushButton_power_control_analog_1->setEnabled(config->get_support_power_control_modes() && config->get_support_AIS_mode());
	ui->pushButton_power_control_amplifier_1->setEnabled(config->get_support_power_control_modes() && config->get_support_amplifier_mode());
	ui->pushButton_power_control_amplifier_1->setVisible(config->get_support_power_control_modes() && config->get_support_amplifier_mode());
	ui->pushButton_power_control_PID_1->setVisible(config->get_support_power_control_modes() && config->get_support_PID_control());
	ui->pushButton_power_control_PID_1->setEnabled(config->get_support_power_control_modes() && config->get_support_PID_control());
	ui->pushButton_power_control_feedforward_1->setVisible(config->get_support_power_control_modes() && config->get_support_feedforward_control());
	ui->pushButton_power_control_feedforward_1->setEnabled(config->get_support_power_control_modes() && config->get_support_feedforward_control());
	ui->label_power_control_mode_1->setVisible(config->get_support_power_control_modes());
	ui->label_power_control_mode_1->setEnabled(config->get_support_power_control_modes());

	ui->pushButton_external_triggering_OFF_1->setVisible(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_OFF_1->setEnabled(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_ON_1->setVisible(config->get_support_external_triggering_mode());
	ui->pushButton_external_triggering_ON_1->setEnabled(config->get_support_external_triggering_mode());
	ui->External_triggering_label->setVisible(config->get_support_external_triggering_mode());
	ui->External_triggering_label->setEnabled(config->get_support_external_triggering_mode());

	ui->pushButton_data_logging_ON_1->setVisible(config->get_support_logging());
	ui->pushButton_data_logging_ON_1->setEnabled(config->get_support_logging());
	ui->pushButton_data_logging_OFF_1->setVisible(config->get_support_logging());
	ui->pushButton_data_logging_OFF_1->setEnabled(config->get_support_logging());
	ui->label_data_logging_1->setVisible(config->get_support_logging());
	ui->label_data_logging_1->setEnabled(config->get_support_logging());

	ui->menu_home_Button->setVisible(config->get_menu_home_enabled());				//enable or disable the home menu
	ui->menu_home_Button->setEnabled(config->get_menu_home_enabled());
	ui->menu_splitter_Button->setVisible(config->get_menu_splitter_enabled());		//enable or disable the splitter menu
	ui->menu_splitter_Button->setEnabled(config->get_menu_splitter_enabled());
	ui->menu_DLL_Button->setVisible(config->get_menu_DLL_enabled());				//enable or disable the DLL menu
	ui->menu_DLL_Button->setEnabled(config->get_menu_DLL_enabled());
	ui->menu_sweep_Button->setVisible(config->get_menu_sweep_enabled());			//enable or disable the Sweep menu
	ui->menu_sweep_Button->setEnabled(config->get_menu_sweep_enabled());
	ui->menu_help_Button->setVisible(config->get_menu_help_enabled());				//enable or disable the Help menu
	ui->menu_help_Button->setEnabled(config->get_menu_help_enabled());
	ui->menu_PSU_Button->setVisible(config->get_menu_PSU_enabled());				//enable or disable the PSU menu
	ui->menu_PSU_Button->setEnabled(config->get_menu_PSU_enabled());
	ui->menu_settings_Button->setVisible(config->get_menu_settings_enabled());		//enable or disable the settings menu
	ui->menu_settings_Button->setEnabled(config->get_menu_settings_enabled());
	ui->menu_power_Button->setVisible(config->get_menu_power_enabled());			//enable or disable the power menu
	ui->menu_power_Button->setEnabled(config->get_menu_power_enabled());

	ui->pushButton_CW_enable_1->setVisible(config->get_controls_CW_enabled());		//enable or disable CW button
	ui->pushButton_CW_enable_1->setEnabled(config->get_controls_CW_enabled());
	ui->pushButton_PWM_enable_1->setVisible(config->get_controls_PWM_enabled());	//enable or disable PWM button
	ui->pushButton_PWM_enable_1->setEnabled(config->get_controls_PWM_enabled());
	ui->pushButton_DLL_enable_1->setVisible(config->get_controls_DLL_enabled());	//enable or disable DLL button
	ui->pushButton_DLL_enable_1->setEnabled(config->get_controls_DLL_enabled());

	/* Show/Hide the Offset button */
	ui->label_power_offset_dB_1->setVisible(config->get_support_power_offset());
	ui->label_power_offset_dB_1->setEnabled(config->get_support_power_offset());
	ui->pushButton_power_offset_dB_1->setVisible(config->get_support_power_offset());
	ui->pushButton_power_offset_dB_1->setEnabled(config->get_support_power_offset());

	/* Hide PSU buttons as necessary */
//	ui->pushButton_PSU_enable_1->setVisible(config->get_support_PSU_controls_enable());
	ui->pushButton_PSU_voltage_1->setVisible(config->get_support_PSU_controls_voltage());
	ui->pushButton_PSU_current_limit_1->setVisible(config->get_support_PSU_controls_current_limit());
	ui->label_PSU_enable_1->setVisible(config->get_support_PSU_controls_enable());
	ui->label_PSU_voltage_1->setVisible(config->get_support_PSU_controls_voltage());
	ui->label_PSU_current_limit_1->setVisible(config->get_support_PSU_controls_current_limit());

	//
	// TODO:
	// This PSU gridlayout management relies too much on perfect content arrangements for the PSU readings layout.
	// Any changes to it in the future are bound to break this function, so it would be good to make it work by disabling/enabling ui elements by name instead of coordinate
	//
	/* Disable PSU labels in the gridLayout based on PSU count */
	for (int i = ui->gridLayout_PSU_readings_2->columnCount() - 1; i >= 0; i--)	//columns / x-axis
	{
		for (int j = 0; j <= 4; j++)	//rows / y-axis
		{
			if (i >= config->get_PSU_count())
			{
				ui->gridLayout_PSU_readings_2->itemAtPosition(j,i)->widget()->setVisible(false);
			}
		}
	}

	/* Create Numpad object + connect input confirmation with the respective handler */
	numpad = new Numpad(5,4, ui->numpadLabel_display, ui->numpadLabel_unit, ui->numpadButton_period, ui->numpadButton_sign);
	connect(numpad, &Numpad::value_confirmed_signal, this, &GUI_1channel::numpad_value_confirmed_handler);

	/* optionally enable the temperature measurements */
	ui->label_temperature_1->setVisible(config->get_read_temperature() >= 1);
	ui->label_temperature_2->setVisible(config->get_read_temperature() >= 1);
	ui->label_temperature_3->setVisible(config->get_read_temperature() >= 1);

	// Help Menu items
	listWidgetItem_system.setText("System");
	listWidgetItem_about.setText("About");
	listWidgetItem_license.setText("License");

	ui->listWidget->addItem(&listWidgetItem_system);
	ui->listWidget->addItem(&listWidgetItem_about);
	ui->listWidget->addItem(&listWidgetItem_license);

	/* If the Help menu is visible, the bottom line of the Settings menu is redudant and therefor hidden out of sight. */
	ui->GUI_version_label_1->setVisible(!config->get_menu_help_enabled());
	ui->GUI_version_label_2->setVisible(!config->get_menu_help_enabled());
	ui->firmware_version_label_1->setVisible(!config->get_menu_help_enabled());
	ui->firmware_version_label_2->setVisible(!config->get_menu_help_enabled());
	ui->SG_board_identity_label_1->setVisible(!config->get_menu_help_enabled());
	ui->SG_board_identity_label_2->setVisible(!config->get_menu_help_enabled());


	/* Load a stylesheet */
	QFile stylesheetfile(QCoreApplication::applicationDirPath() + "/theme.css");
	QString style;
	stylesheetfile.open(QFile::ReadOnly);
	style = stylesheetfile.readAll();
	stylesheetfile.close();

	/* Set stylesheet across the board */
	this->setStyleSheet(style);

	/* This is necessary so the logo image loads correctly regardless of the working path. Image needs to be in the same folder as the executable. */
	ui->logo_label->setStyleSheet("image: url(" + QCoreApplication::applicationDirPath() + "/logo.png);\nimage-position: top;\n");
	//
	// TODO:
	// The application path is interpretted as CSS code. If the path contains an unclosed set of bracket for example, that can mess with the CSS code and result in the Logo not displaying correctly.
	// Apparently this is also a minor security risk, as the filepath is execute as css code, though it doesn't matter much in a GUI like this.
	// Either the directory path needs to be sanitized (like escaping all brackets / special characters), or a relative path should be used instead.
	//
	// TODO:
	// Can this just be handled from the css file somehow?
	//


	/* Indefinite disables */
	// Disable the Slave End button - Not a very useful feature

	// Disable some aesthetic labels in the Shutdown / Hamburger Menu - There used to be more buttons, but they've been cleaned out and the extra labels were making things too busy. */
	ui->label_menu_power_spacer_1->setVisible(false);
	ui->label_menu_power_spacer_5->setVisible(false);

	//	/* If Linux OS + RPi HAT (ideally Raspberry Pi) -> enable interlock label and set correct string
	//	 * If Windows -> disable the interlock label */
	//#if defined(Q_OS_LINUX)
	//	ui->label_interlock_1->setVisible(config->get_support_HAT_B14_0835());
	//	ui->label_interlock_1->setEnabled(config->get_support_HAT_B14_0835());
	//	ui->label_interlock_1->setText("Power to modules is DISABLED.");
	//#elif defined(Q_OS_WINDOWS)
		ui->label_interlock_1->setVisible(false);
	//#endif

	#ifndef DEBUG
	ui->pushButton_SWP_save_sweep_1->setVisible(false);
	#elif DEBUG
	show_notification("THIS IS A DEBUG BUILD!");
	ui->settings_plainTextEdit->appendPlainText(">\tTHIS IS A DEBUG BUILD!");
	#endif

}	//End constructor


GUI_1channel::~GUI_1channel()
{
	qDebug() << "Exiting Normally";
	RF_system_constructor->Remote_Command_Mode->RCM_kill();
	RF_system_constructor->~RF_System_1channel();
	delete ui;
}

/**********************************************************************************************************************************************************************************
 * UTILITY FUNCTIONS
 * *******************************************************************************************************************************************************************************/
/* Enable or disable automatic error clearing */
void GUI_1channel::configure_auto_error_clearing()
{
	//
	// TODO:
	// It's a little bit iffy that button states are used for a check, instead of something more proper like enable states for RCM and Modbus OFF.
	// If these were SG commands we could poll the Get command. Maybe we need something similar internally for consistency and safety's sake?
	//

	//
	// TODO:
	// This code can only function properly because the Button states are configured to OFF by default through the UI editor.
	// But the code should be more robust than that. It can't rely on the default state of the UI to perform it's job properly!
	//

	bool enable = true;
	if (RCM_supported == true)
	{
		enable &= RCM_error_clearing;
	}

	if (config->get_support_modbus_mode())
	{
		enable &= ui->pushButton_modbus_OFF_1->isChecked();
	}

	if (config->get_support_ethercat_mode())
	{
		enable &= ui->pushButton_etherCAT_OFF_1->isChecked();
	}
	
	if (config->get_error_clear_mode() != 1)
	{
		enable &= false;
	}

	emit signal_set_error_clearing_enable(1, enable);
}


void GUI_1channel::handler_splitter_configuration(int intrasys_num, int splitter_mode, int splitter_channel_count)
{
	(void)intrasys_num;

	/* Splitter Mode 1 and 3 support the splitter subchannel functionality, other modes do not --> Check for odd values of splitter_mode to enable controls.
	 * Relying on even/odd numbers is a bit of a gamble, but with some luck this might scale in the future */
	bool support_controls = false;

	if ((splitter_mode % 2) > 0)
	{
		support_controls = true;

		/* Only connect the NChannel PCS/MCS/GCS controls if the splitter mode is sensible and the respective config settings are enabled. */
		for (int i = 0; i < RF_System::Channels->count(); i++)
		{
			if (config->get_support_splitter_controls_phase())
			{
				connect(this, &GUI_1channel::signal_setPhase_NChannel, RF_System::Channels->at(i), &RF_Channel::Set_Phase_NChannel);
				connect(RF_System::Channels->at(i), &RF_Channel::signal_NChannel_phase_get, this, &GUI_1channel::handler_phase_NChannel_get);
			}

			if (config->get_support_splitter_controls_magnitude())
			{
				connect(this, &GUI_1channel::signal_setMagnitude_NChannel, RF_System::Channels->at(i), &RF_Channel::Set_Magnitude_NChannel);
				connect(RF_System::Channels->at(i), &RF_Channel::signal_NChannel_magnitude_get, this, &GUI_1channel::handler_magnitude_NChannel_get);
			}

			if (config->get_support_splitter_controls_attenuation())
			{
				connect(this, &GUI_1channel::signal_setAttenuation_NChannel, RF_System::Channels->at(i), &RF_Channel::Set_Attenuation_NChannel);
				connect(RF_System::Channels->at(i), &RF_Channel::signal_NChannel_attenuation_get, this, &GUI_1channel::handler_attenuation_NChannel_get);
			}
		}
	}

	/* Show/Hide the Attenuation, Magnitude and Phase controls of the splitter
	 * Due to UI space issues, the controls are only supported for up to 4 channels. If 5+ channels are configured only the FWD/RFL readings are shown. */
	for (int i = 0; i < frame_list.count(); i++)
	{
		splitter_button_magnitude_list.at(i)->setVisible(splitter_channel_count <= 4 && config->get_support_splitter_controls_magnitude() && support_controls);
		splitter_button_magnitude_list.at(i)->setEnabled(splitter_channel_count <= 4 && config->get_support_splitter_controls_magnitude() && support_controls);
		splitter_label_magnitude_list.at(i)->setVisible(splitter_channel_count <= 4 && config->get_support_splitter_controls_magnitude() && support_controls);
		splitter_label_magnitude_list.at(i)->setEnabled(splitter_channel_count <= 4 && config->get_support_splitter_controls_magnitude() && support_controls);

		splitter_button_attenuation_list.at(i)->setVisible(splitter_channel_count <= 4 && config->get_support_splitter_controls_attenuation() && support_controls);
		splitter_button_attenuation_list.at(i)->setEnabled(splitter_channel_count <= 4 && config->get_support_splitter_controls_attenuation() && support_controls);
		splitter_label_attenuation_list.at(i)->setVisible(splitter_channel_count <= 4 && config->get_support_splitter_controls_attenuation() && support_controls);
		splitter_label_attenuation_list.at(i)->setEnabled(splitter_channel_count <= 4 && config->get_support_splitter_controls_attenuation() && support_controls);

		splitter_button_phase_list.at(i)->setVisible(splitter_channel_count <= 4 && config->get_support_splitter_controls_phase() && support_controls);
		splitter_button_phase_list.at(i)->setEnabled(splitter_channel_count <= 4 && config->get_support_splitter_controls_phase() && support_controls);
		splitter_label_phase_list.at(i)->setVisible(splitter_channel_count <= 4 && config->get_support_splitter_controls_phase() && support_controls);
		splitter_label_phase_list.at(i)->setEnabled(splitter_channel_count <= 4 && config->get_support_splitter_controls_phase() && support_controls);
	}

	/* Hide the unused splitter channel frames */
	for (int i = frame_list.count() ; i > splitter_channel_count; i--)
	{
		frame_list.at(i-1)->setEnabled(false);
		frame_list.at(i-1)->setVisible(false);
	}
}

void GUI_1channel::handler_channel_initialization_complete(int intrasys_num, bool success)
{
	if (success == true)
	{
		/* Set the start condition for Logging */
		if (config->get_support_logging())
		{
			handler_datalogging(config->get_logging_enabled());
		}

		/* Set the start condition for Modbus */
		handler_modbus_mode();

		/* Set the start condition for EtherCAT */
		handler_ethercat_mode();


		/* Configure Power Supply start state */
		//
		// TODO:
		// PSU_reset is a workaround for the lack of a startup script feature.
		// Delete when possible
		//
		PSU_reset();

		/* Set the start condition for RCM. */
		handler_RCM_mode_startup();

		/* Select the system 'tab' in the Help menu by default */
		ui->listWidget->setCurrentItem(&listWidgetItem_system);

		qDebug() << "\n\nGUI: initialization complete.";
		qDebug() << "============================\n";
	}
	else if (success == false)
	{
		qDebug() << "\n\nGUI: Subsystem[" << intrasys_num << "] Initialization failed.";
		qDebug() << "============================\n";
		if (RF_system_constructor->config->get_init_failed_exit() == true)
		{
			qDebug() << "Exiting GUI.\n";
			this->close();
		}
	}
}

//Miscellaneous
void GUI_1channel::handler_channel_working(int intrasys_number, bool enabled)
{
	(void)intrasys_number;

	this->setEnabled(enabled);
}

/**********************************************************************************************************************************************************************************
 * Parameter buttons
 * *******************************************************************************************************************************************************************************/

// HOME
void GUI_1channel::on_pushButton_VGA_attenuation_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->VGA_Attenuation();
	numpad_parameter_select(ui->pushButton_VGA_attenuation_1, &numpad_value, "dB", 1, 1, true, config->get_support_negative_attenuation());
}
void GUI_1channel::on_pushButton_IQMod_magnitude_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->IQMod_Magnitude();
	numpad_parameter_select(ui->pushButton_IQMod_magnitude_1, &numpad_value, "%", 1);
}
void GUI_1channel::on_pushButton_power_watt_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_1, &numpad_value, "watt", 1);
}
void GUI_1channel::on_pushButton_power_dbm_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_dBm();
	numpad_parameter_select(ui->pushButton_power_dbm_1, &numpad_value, "dBm", 1);
}
void GUI_1channel::on_pushButton_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Frequency();
	numpad_parameter_select(ui->pushButton_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_power_offset_dB_1_clicked()
{	numpad_value = RF_System::Channels->at(0)->Offset_dB();
	numpad_parameter_select(ui->pushButton_power_offset_dB_1, &numpad_value, "dB", 3, 1, true, true);
}
void GUI_1channel::on_pushButton_phase_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase();
	numpad_parameter_select(ui->pushButton_phase_1, &numpad_value, "°", 0, 1 , false);
}
void GUI_1channel::on_pushButton_PWM_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PWM_Frequency();
	numpad_parameter_select(ui->pushButton_PWM_frequency_1, &numpad_value, "Hz", 0, 1, false);
}
void GUI_1channel::on_pushButton_PWM_duty_cycle_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->PWM_Duty_Cycle();
	numpad_parameter_select(ui->pushButton_PWM_duty_cycle_1, &numpad_value, "%", 0, 1, false);
}
void GUI_1channel::on_pushButton_DLL_step_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Step_Frequency();
	numpad_parameter_select(ui->pushButton_DLL_step_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_DLL_threshold_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Threshold();
	numpad_parameter_select(ui->pushButton_DLL_threshold_1, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_ALL_threshold_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Threshold();
	numpad_parameter_select(ui->pushButton_ALL_threshold_1, &numpad_value, "V");
}

// Splitter/NChannel
void GUI_1channel::on_pushButton_splitter_phase_CH1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(0);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH1, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(1);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH2, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH3_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(2);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH3, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH4_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(3);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH4, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH5_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(4);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH5, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH6_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(5);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH6, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH7_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(6);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH7, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_phase_CH8_clicked(){
	numpad_value = RF_System::Channels->at(0)->Phase_NChannel(7);
	numpad_parameter_select(ui->pushButton_splitter_phase_CH8, &numpad_value, "°");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(0);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH1, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(1);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH2, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH3_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(2);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH3, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH4_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(3);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH4, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH5_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(4);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH5, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH6_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(5);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH6, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH7_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(6);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH7, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_attenuation_CH8_clicked(){
	numpad_value = RF_System::Channels->at(0)->Attenuation_NChannel(7);
	numpad_parameter_select(ui->pushButton_splitter_attenuation_CH8, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH1_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(0);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH1, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(1);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH2, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH3_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(2);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH3, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH4_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(3);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH4, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH5_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(4);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH5, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH6_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(5);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH6, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH7_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(6);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH7, &numpad_value, "%");
}
void GUI_1channel::on_pushButton_splitter_magnitude_CH8_clicked(){
	numpad_value = RF_System::Channels->at(0)->Magnitude_NChannel(7);
	numpad_parameter_select(ui->pushButton_splitter_magnitude_CH8, &numpad_value, "%");
}

// ALL
void GUI_1channel::on_pushButton_ALL_frequency_limit_upper_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Frequency_Limit_Upper();
	numpad_parameter_select(ui->pushButton_ALL_frequency_limit_upper_1, &numpad_value, "V");
}
void GUI_1channel::on_pushButton_ALL_frequency_limit_lower_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Frequency_Limit_Lower();
	numpad_parameter_select(ui->pushButton_ALL_frequency_limit_lower_1, &numpad_value, "V");
}
void GUI_1channel::on_pushButton_ALL_threshold_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->ALL_Threshold();
	numpad_parameter_select(ui->pushButton_ALL_threshold_2, &numpad_value, "V");
}
void GUI_1channel::on_pushButton_DVGA_attenuation_forward_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DVGA_Attenuation_Forward();
	numpad_parameter_select(ui->pushButton_DVGA_attenuation_forward_1, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_DVGA_attenuation_reflected_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DVGA_Attenuation_Reflected();
	numpad_parameter_select(ui->pushButton_DVGA_attenuation_reflected_1, &numpad_value, "dB");
}

void GUI_1channel::on_pushButton_ALL_reference_open_clicked()
{
	emit signal_RF_enable(1,false);
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->ALL_reference_page));
}

void GUI_1channel::on_pushButton_DVGA_amplifier_forward_1_clicked()
{
	emit signal_setDVGA_amplifier_forward_enable(1, ui->pushButton_DVGA_amplifier_forward_1->isChecked());
}

void GUI_1channel::on_pushButton_DVGA_amplifier_reflected_1_clicked()
{
	emit signal_setDVGA_amplifier_reflected_enable(1, ui->pushButton_DVGA_amplifier_reflected_1->isChecked());
}

/* The button on the reference page that sends you back to the main */
void GUI_1channel::on_pushButton_ALL_reference_back_clicked()
{
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
}

// SWEEP
void GUI_1channel::on_pushButton_SWP_start_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Start_Frequency();
	numpad_parameter_select(ui->pushButton_SWP_start_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_SWP_stop_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Stop_Frequency();
	numpad_parameter_select(ui->pushButton_SWP_stop_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_SWP_step_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Step_Frequency();
	numpad_parameter_select(ui->pushButton_SWP_step_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_SWP_power_dbm_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Power_dBm();
	numpad_parameter_select(ui->pushButton_SWP_power_dbm_1, &numpad_value, "dBm", 1);
}
void GUI_1channel::on_pushButton_SWP_power_watt_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->SWP_Power_watt();
	numpad_parameter_select(ui->pushButton_SWP_power_watt_1, &numpad_value, "watt", 1);
}

// DLL
void GUI_1channel::on_pushButton_DLL_frequency_limit_lower_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Frequency_Limit_Lower();
	numpad_parameter_select(ui->pushButton_DLL_frequency_limit_lower_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_DLL_frequency_limit_upper_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Frequency_Limit_Upper();
	numpad_parameter_select(ui->pushButton_DLL_frequency_limit_upper_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_DLL_start_frequency_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Start_Frequency();
	numpad_parameter_select(ui->pushButton_DLL_start_frequency_1, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_DLL_step_frequency_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Step_Frequency();
	numpad_parameter_select(ui->pushButton_DLL_step_frequency_2, &numpad_value, "MHz", 3);
}
void GUI_1channel::on_pushButton_DLL_threshold_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Threshold();
	numpad_parameter_select(ui->pushButton_DLL_threshold_2, &numpad_value, "dB");
}
void GUI_1channel::on_pushButton_DLL_main_delay_1_clicked(){
	numpad_value = RF_System::Channels->at(0)->DLL_Main_Delay();
	numpad_parameter_select(ui->pushButton_DLL_main_delay_1, &numpad_value, "ms", 0, 1, false);
}
void GUI_1channel::on_pushButton_power_dbm_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_dBm();
	numpad_parameter_select(ui->pushButton_power_dbm_2, &numpad_value, "dBm", 1);
}
void GUI_1channel::on_pushButton_power_watt_2_clicked(){
	numpad_value = RF_System::Channels->at(0)->Power_watt();
	numpad_parameter_select(ui->pushButton_power_watt_2, &numpad_value, "watt", 1);
}

//PSU
void GUI_1channel::on_pushButton_PSU_voltage_1_clicked()
{
	numpad_value = RF_System::Channels->at(0)->PSU_Voltage_Setpoint();
	/* PSU_Voltage_Setpoint may return a negative value to indicate something is wrong; however this results in unintentionally inputting negative values through the numpad */
	if (numpad_value < 0)
	{
		numpad_value = 0;
	}
	numpad_parameter_select(ui->pushButton_PSU_voltage_1, &numpad_value, "V", 1, 1, false);
}

void GUI_1channel::on_pushButton_PSU_current_limit_1_clicked()
{
	numpad_value = RF_System::Channels->at(0)->PSU_Current_Limit();
	/* PSU_Current_Limit may return a negative value to indicate something is wrong; however this results in unintentionally inputting negative values through the numpad */
	if (numpad_value < 0)
	{
		numpad_value = 0;
	}
	numpad_parameter_select(ui->pushButton_PSU_current_limit_1, &numpad_value, "A", 1, 1, false);
}

void GUI_1channel::on_pushButton_PSU_enable_1_clicked(bool checked)
{
	emit signal_setPSU_enable(1, checked);
}

/**********************************************************************************************************************************************************************************
 * NUMPAD BUTTONS
 * *******************************************************************************************************************************************************************************/

void GUI_1channel::on_numpadButton_0_clicked(){
	numpad->press_0();
}
void GUI_1channel::on_numpadButton_1_clicked(){
	numpad->press_1();
}
void GUI_1channel::on_numpadButton_2_clicked(){
	numpad->press_2();
}
void GUI_1channel::on_numpadButton_3_clicked(){
	numpad->press_3();
}
void GUI_1channel::on_numpadButton_4_clicked(){
	numpad->press_4();
}
void GUI_1channel::on_numpadButton_5_clicked(){
	numpad->press_5();
}
void GUI_1channel::on_numpadButton_6_clicked(){
	numpad->press_6();
}
void GUI_1channel::on_numpadButton_7_clicked(){
	numpad->press_7();
}
void GUI_1channel::on_numpadButton_8_clicked(){
	numpad->press_8();
}
void GUI_1channel::on_numpadButton_9_clicked(){
	numpad->press_9();
}
void GUI_1channel::on_numpadButton_period_clicked(){
	numpad->press_period();
}
void GUI_1channel::on_numpadButton_backspace_clicked(){
	numpad->press_delete();
}
void GUI_1channel::on_numpadButton_clear_all_clicked(){
	numpad->press_delete_all();
}
void GUI_1channel::on_numpadButton_arrow_left_clicked(){
	numpad->press_indicator_left();
}
void GUI_1channel::on_numpadButton_arrow_right_clicked(){
	numpad->press_indicator_right();
}
void GUI_1channel::on_numpadButton_plus_clicked(){
	numpad->press_plus();
}
void GUI_1channel::on_numpadButton_minus_clicked(){
	numpad->press_minus();
}
void GUI_1channel::on_numpadButton_sign_clicked(){
	numpad->press_sign();
}
void GUI_1channel::on_numpadButton_ok_clicked(){
	numpad->press_confirm_input();
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
}

/**********************************************************************************************************************************************************************************
 * FUNCTIONS FOR NUMPAD
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::numpad_parameter_select(QPushButton *button, double *parameter, QString unit, int precision, double scale,  bool period_enable, bool sign_enable)
{
	numpad->setOutputButton(button);
	numpad->load_value(parameter, unit, period_enable, sign_enable, scale);
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_numpad));
	scale_multiplier = scale;
	this->precision = precision;
}

void GUI_1channel::numpad_value_confirmed_handler(QPushButton *button, double *num)
{
//	ok_check_minmax(button);
	QString str = zeroChopper(QString::number(*num * scale_multiplier,'f', precision));
	button->setText(str);

	//Home Page parameters
	if (button == ui->pushButton_frequency_1){
		emit signal_setFrequency(1, *num);
	}
	else if (button == ui->pushButton_power_dbm_1){
		emit signal_setPowerdBm(1, *num);
	}
	else if (button == ui->pushButton_power_watt_1){
		emit signal_setPowerWatt(1, *num);
	}
	else if (button == ui->pushButton_VGA_attenuation_1){
		emit signal_setVGA_attenuation(1, *num);
	}
	else if (button == ui->pushButton_IQMod_magnitude_1){
		emit signal_setIQMod_magnitude(1, *num);
	}
	else if (button == ui->pushButton_phase_1){
		emit signal_setPhase(1, *num);
	}
	else if (button == ui->pushButton_PWM_frequency_1){
		emit signal_setPWM_frequency(1,*num);
	}
	else if (button == ui->pushButton_PWM_duty_cycle_1){
		emit signal_setPWM_duty_cycle(1,*num);
	}
	else if (button == ui->pushButton_DLL_step_frequency_1){
		emit signal_setDLL_step_frequency(1,*num);
	}
	else if (button == ui->pushButton_DLL_threshold_1){
		emit signal_setDLL_threshold(1,*num);
	}
	else if (button == ui->pushButton_ALL_threshold_1){
		emit signal_setALL_threshold(1,*num);
	}
	else if (button == ui->pushButton_power_offset_dB_1){
		emit signal_setOffsetdB(1, *num);
	}

	// Splitter / NChannel Page paremeters
	else if (button == ui->pushButton_splitter_phase_CH1){
		emit signal_setPhase_NChannel(1, 1, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH2){
		emit signal_setPhase_NChannel(1, 2, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH3){
		emit signal_setPhase_NChannel(1, 3, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH4){
		emit signal_setPhase_NChannel(1, 4, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH5){
		emit signal_setPhase_NChannel(1, 5, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH6){
		emit signal_setPhase_NChannel(1, 6, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH7){
		emit signal_setPhase_NChannel(1, 7, *num);
	}
	else if (button == ui->pushButton_splitter_phase_CH8){
		emit signal_setPhase_NChannel(1, 8, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH1){
		emit signal_setAttenuation_NChannel(1, 1, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH2){
		emit signal_setAttenuation_NChannel(1, 2, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH3){
		emit signal_setAttenuation_NChannel(1, 3, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH4){
		emit signal_setAttenuation_NChannel(1, 4, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH5){
		emit signal_setAttenuation_NChannel(1, 5, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH6){
		emit signal_setAttenuation_NChannel(1, 6, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH7){
		emit signal_setAttenuation_NChannel(1, 7, *num);
	}
	else if (button == ui->pushButton_splitter_attenuation_CH8){
		emit signal_setAttenuation_NChannel(1, 8, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH1){
		emit signal_setMagnitude_NChannel(1, 1, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH2){
		emit signal_setMagnitude_NChannel(1, 2, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH3){
		emit signal_setMagnitude_NChannel(1, 3, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH4){
		emit signal_setMagnitude_NChannel(1, 4, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH5){
		emit signal_setMagnitude_NChannel(1, 5, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH6){
		emit signal_setMagnitude_NChannel(1, 6, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH7){
		emit signal_setMagnitude_NChannel(1, 7, *num);
	}
	else if (button == ui->pushButton_splitter_magnitude_CH8){
		emit signal_setMagnitude_NChannel(1, 8, *num);
	}

	//ALL Page parameters
	else if (button == ui->pushButton_ALL_frequency_limit_upper_1){
		emit signal_setALL_frequency_limit_upper(1,*num);
	}
	else if (button == ui->pushButton_ALL_frequency_limit_lower_1){
		emit signal_setALL_frequency_limit_lower(1,*num);
	}
	else if (button == ui->pushButton_ALL_threshold_2){
		emit signal_setALL_threshold(1,*num);
	}
	else if (button == ui->pushButton_DVGA_attenuation_forward_1){
		emit signal_setDVGA_attentuation_forward(1,*num);
	}
	else if (button == ui->pushButton_DVGA_attenuation_reflected_1){
		emit signal_setDVGA_attentuation_reflected(1,*num);
	}

	//DLL Page parameters
	else if (button == ui->pushButton_DLL_frequency_limit_upper_1){
		emit signal_setDLL_frequency_limit_upper(1,*num);
	}
	else if (button == ui->pushButton_DLL_frequency_limit_lower_1){
		emit signal_setDLL_frequency_limit_lower(1,*num);
	}
	else if (button == ui->pushButton_DLL_start_frequency_1){
		emit signal_setDLL_start_frequency(1,*num);
	}
	else if (button == ui->pushButton_DLL_step_frequency_2){
		emit signal_setDLL_step_frequency(1,*num);
	}
	else if (button == ui->pushButton_DLL_threshold_2){
		emit signal_setDLL_threshold(1,*num);
	}
	else if (button == ui->pushButton_DLL_main_delay_1){
		emit signal_setDLL_main_delay(1,*num);
	}
	else if (button == ui->pushButton_power_dbm_2){
		emit signal_setPowerdBm(1, *num);
	}
	else if (button == ui->pushButton_power_watt_2){
		emit signal_setPowerWatt(1, *num);
	}

	//Sweep Page parameters
	else if (button == ui->pushButton_SWP_start_frequency_1){
		emit signal_setSWP_start_frequency(1,*num);
	}
	else if (button == ui->pushButton_SWP_stop_frequency_1){
		emit signal_setSWP_stop_frequency(1,*num);
	}
	else if (button == ui->pushButton_SWP_step_frequency_1){
		emit signal_setSWP_step_frequency(1,*num);
	}
	else if (button == ui->pushButton_SWP_power_dbm_1){
		emit signal_setSWP_power_dbm(1,*num);
	}
	else if (button == ui->pushButton_SWP_power_watt_1){
		emit signal_setSWP_power_watt(1,*num);
	}

	//PSU Page parameters
	else if (button == ui->pushButton_PSU_voltage_1)
	{
		emit signal_setPSU_voltage(1,*num);
	}
	else if (button == ui->pushButton_PSU_current_limit_1)
	{
		emit signal_setPSU_current_limit(1,*num);
	}

}


/**********************************************************************************************************************************************************************************
 * MENU BUTTONS
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_menu_home_Button_clicked()
{
	menuButton_manager(ui->menu_home_Button);
}
void GUI_1channel::on_menu_splitter_Button_clicked()
{
	menuButton_manager(ui->menu_splitter_Button);
}
void GUI_1channel::on_menu_ALL_Button_clicked()
{
	menuButton_manager(ui->menu_ALL_Button);
}
void GUI_1channel::on_menu_settings_Button_clicked()
{
	menuButton_manager(ui->menu_settings_Button);
}
void GUI_1channel::on_menu_sweep_Button_clicked()
{
	menuButton_manager(ui->menu_sweep_Button);
}
void GUI_1channel::on_menu_help_Button_clicked()
{
	menuButton_manager(ui->menu_help_Button);
}
void GUI_1channel::on_menu_DLL_Button_clicked()
{
	menuButton_manager(ui->menu_DLL_Button);
}
void GUI_1channel::on_menu_PSU_Button_clicked()
{
	menuButton_manager(ui->menu_PSU_Button);
}
void GUI_1channel::on_menu_power_Button_clicked()
{
	menuButton_manager(ui->menu_power_Button);
}

void GUI_1channel::menuButton_manager(QPushButton *menuButton_pointer)
{
	/* HOME page */
	if (menuButton_pointer == ui->menu_home_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->home_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* N-Channel / Splitter page */
	if (menuButton_pointer == ui->menu_splitter_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->splitter_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* ALL page */
	if (menuButton_pointer == ui->menu_ALL_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->ALL_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* SETTINGS page */
	if (menuButton_pointer == ui->menu_settings_Button)
	{
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Settings_page));
		emit signal_RF_enable(1,false);			//disable RF power because user loses sight of the active data.
	}

	/* SWEEP page */
	if (menuButton_pointer == ui->menu_sweep_Button)
	{
		if (ui->pushButton_SWP_execute_1->isChecked() == true)
		{
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Sweep_graph_page));
			emit signal_RF_enable(1,false);		//disable RF power because user loses sight of the active data.
		}
		else
		{
			ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->sweep_controls_page));
			ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
			ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		}

	}

	/* HELP page */
	if (menuButton_pointer == ui->menu_help_Button)
	{
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Help_page));
		emit signal_RF_enable(1,false);			//disable RF power because user loses sight of the active data.
	}

	/* DLL page */
	if (menuButton_pointer == ui->menu_DLL_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->DLL_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* PSU page */
	if (menuButton_pointer == ui->menu_PSU_Button)
	{
		ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->PSU_controls_page));
		ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
	}

	/* POWER / SHUTDOWN page */
	if (menuButton_pointer == ui->menu_power_Button)
	{
		emit signal_RF_enable(1,false);			//disable RF power because user loses sight of the active data.
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_page));
		last_page_index = ui->stackedWidget_3->currentIndex();		//used by the Exit_No_button to return to this page.
	}

	/* Update the visiblity of the execute SWP button:
	 * Should only be shown of the current page is the Sweep page, and the configuration allows for sweeps.
	 * This function call needs to be at the end of the function to ensure all the stackedWidget page indexes are updated. */
	show_sweep_button();
}


/**********************************************************************************************************************************************************************************
 * UTILITY BUTTONS
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_pushButton_RF_enable_1_clicked()
{
	emit signal_RF_enable(1, ui->pushButton_RF_enable_1->isChecked());
}

void GUI_1channel::on_pushButton_unit_power_reflected_1_clicked()
{
	set_power_reading_units_watt(ui->pushButton_unit_power_reflected_1->isChecked());
}

void GUI_1channel::on_pushButton_unit_power_forward_1_clicked()
{
	set_power_reading_units_watt(ui->pushButton_unit_power_forward_1->isChecked());
}

void GUI_1channel::on_pushButton_unit_S11_1_clicked()
{
	set_power_reading_units_watt(ui->pushButton_unit_S11_1->isChecked());
}

void GUI_1channel::set_power_reading_units_watt(bool mode)
{
	if(mode == false)
	{
		ui->pushButton_unit_power_reflected_1->setText("dBm");
		ui->pushButton_unit_power_forward_1->setText("dBm");
		ui->pushButton_unit_S11_1->setText("dB");
		ui->S11_title_label->setText("S11:");
		ui->pushButton_unit_S11_1->setChecked(false);
		ui->pushButton_unit_power_forward_1->setChecked(false);
		ui->pushButton_unit_power_reflected_1->setChecked(false);
		ui->pushButton_SWP_unit->setText("Unit: log");
		ui->pushButton_SWP_unit->setChecked(false);
	}
	else
	{
		ui->pushButton_unit_power_reflected_1->setText("watt");
		ui->pushButton_unit_power_forward_1->setText("watt");
		ui->pushButton_unit_S11_1->setText("%");
		ui->S11_title_label->setText("Reflection:");
		ui->pushButton_unit_S11_1->setChecked(true);
		ui->pushButton_unit_power_forward_1->setChecked(true);
		ui->pushButton_unit_power_reflected_1->setChecked(true);
		ui->pushButton_SWP_unit->setText("Unit: linear");
		ui->pushButton_SWP_unit->setChecked(true);
	}

	emit signal_request_PA_power_update(1);		//0 = logarithmic, 1 = linear;
}



/**********************************************************************************************************************************************************************************
 * MODE BUTTONS
 * *******************************************************************************************************************************************************************************/

//Maybe phase out the CW button entirely?
void GUI_1channel::on_pushButton_CW_enable_1_clicked()
{
	emit signal_setPWM_enable(1, false);
	emit signal_setALL_enable(1, false);
	emit signal_setDLL_enable(1, false);
}

void GUI_1channel::on_pushButton_PWM_enable_1_clicked()
{
	emit signal_setPWM_enable(1, ui->pushButton_PWM_enable_1->isChecked());
}

void GUI_1channel::on_pushButton_ALL_enable_1_clicked()
{
	emit signal_setALL_enable(1, ui->pushButton_ALL_enable_1->isChecked());
}

void GUI_1channel::on_pushButton_DLL_enable_1_clicked()
{
	emit signal_setDLL_enable(1, ui->pushButton_DLL_enable_1->isChecked());
}

/**********************************************************************************************************************************************************************************
 * REMOTE COMMAND MODE
 **********************************************************************************************************************************************************************************
 * Open an interface through which the user can communicate with the SGx board (through the GUI).
 * - 'Live' RCM: All automated readings continue and the UI remains usable, however only commands with single-line responses are accepted and certain commands are blacklisted. There's a special workaround in place for sweeping.
 *		> USB
 *		> RS-232
 * - 'Blind' RCM: All automated readings are stopped and the UI is disabled, but there are no limitations to the communication.
 *		> USB
 *		> RS-232
 * *******************************************************************************************************************************************************************************/

/* Handle error clearing signal from RCM. RCM indicates it is OK to enable automatic error clearing, however the GUI makes the final call, based on the enable state of other interfaces. */
void GUI_1channel::handler_RCM_error_clearing(int intrasys_num, bool enable)
{
	(void) intrasys_num;
	RCM_error_clearing = enable;
	configure_auto_error_clearing();
}

/* Handler for messages from RCM Class. Informative messages and error indication. */
void GUI_1channel::handler_RCM_notification(QString string)
{
	ui->settings_plainTextEdit->appendPlainText(">\t" + string);
}

/* Set start condition for RCM mode at boot / reset */
void GUI_1channel::handler_RCM_mode_startup()
{
	if (RCM_supported == true)
	{
		switch(config->get_remote_command_mode())
		{
			case RCM_MODE::RCM_MODE_OFF:
				ui->pushButton_remote_command_OFF_1->click();
				break;

			case RCM_MODE::RCM_MODE_TCP_LIVE:
				ui->pushButton_remote_command_TCP_1->click();
				break;

			case RCM_MODE::RCM_MODE_TCP_BLIND:
				ui->menu_settings_Button->click();	//Blind RCM locks user to Setting Menu screen, so need move there first!
				ui->pushButton_remote_command_TCP_blind_1->click();
				break;

			case RCM_MODE::RCM_MODE_USB_LIVE:
				ui->pushButton_remote_command_USB_1->click();
				break;

			case RCM_MODE::RCM_MODE_USB_BLIND:
				ui->menu_settings_Button->click();	//Blind RCM locks user to Setting Menu screen, so need move there first!
				ui->pushButton_remote_command_USB_blind_1->click();
				break;

			case RCM_MODE::RCM_MODE_RS232_LIVE:
				ui->pushButton_remote_command_RS232_1->click();
				break;

			case RCM_MODE::RCM_MODE_RS232_BLIND:
				ui->menu_settings_Button->click();	//Blind RCM locks user to Setting Menu screen, so need move there first!
				ui->pushButton_remote_command_RS232_blind_1->click();
				break;

			default:
				ui->pushButton_remote_command_OFF_1->click();
				break;
		}
	}
}

/* Handle feedback (UI management) about RCM mode from RCM Class */
void GUI_1channel::handler_remote_command_mode_get(RCM_MODE mode)
{
	remote_command_mode = mode;
	switch (mode)
	{
		case RCM_MODE::RCM_MODE_INVALID:
			//
			// TODO:
			// Do something?
			//
			break;

		case RCM_MODE::RCM_MODE_OFF:

			ui->pushButton_remote_command_OFF_1->setChecked(true);
			RCM_controls_enable(true);

			break;

		case RCM_MODE::RCM_MODE_USB_LIVE:

			ui->pushButton_remote_command_USB_1->setChecked(true);
			RCM_controls_enable(true);

			break;

		case RCM_MODE::RCM_MODE_USB_BLIND:

			ui->pushButton_remote_command_USB_blind_1->setChecked(true);
			RCM_controls_enable(false);
			break;

		case RCM_MODE::RCM_MODE_TCP_LIVE:

			ui->pushButton_remote_command_TCP_1->setChecked(true);
			RCM_controls_enable(true);
			break;

		case RCM_MODE::RCM_MODE_TCP_BLIND:

			ui->pushButton_remote_command_TCP_blind_1->setChecked(true);
			RCM_controls_enable(false);
			break;

		case RCM_MODE::RCM_MODE_RS232_LIVE:

			ui->pushButton_remote_command_RS232_1->setChecked(true);
			RCM_controls_enable(true);
			break;

		case RCM_MODE::RCM_MODE_RS232_BLIND:

			ui->pushButton_remote_command_RS232_blind_1->setChecked(true);
			RCM_controls_enable(false);
			break;

		case RCM_MODE::RCM_MODE_RS485_LIVE:

			break;

		case RCM_MODE::RCM_MODE_RS485_BLIND:

			break;

		default:

			break;
	}
}

/* Disable Remote Command Mode. */
void GUI_1channel::on_pushButton_remote_command_OFF_1_clicked()
{
	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_OFF);
}

/* Enable 'Live' USB RCM. */
void GUI_1channel::on_pushButton_remote_command_USB_1_clicked()
{
	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_USB_LIVE);
}

/* Enable 'Blind' USB RCM. */
void GUI_1channel::on_pushButton_remote_command_USB_blind_1_clicked()
{
	/* Disable Modbus during blind RCM */
	if (config->get_support_modbus_mode())
	{
		ui->pushButton_modbus_OFF_1->click();
	}

	/* Disable EtherCAT during blind RCM */
	if (config->get_support_ethercat_mode())
	{
		ui->pushButton_etherCAT_OFF_1->click();
	}

	/* Disable logging during blind RCM */
	if (config->get_support_logging())
	{
		ui->pushButton_data_logging_OFF_1->click();
	}

	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_USB_BLIND);
}

/* Enable 'Live' RS-232 RCM. */
void GUI_1channel::on_pushButton_remote_command_RS232_1_clicked()
{
	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_RS232_LIVE);
}

/* Enable 'Blind' RS-232 RCM. */
void GUI_1channel::on_pushButton_remote_command_RS232_blind_1_clicked()
{
	/* Disable Modbus during blind RCM */
	if (config->get_support_modbus_mode())
	{
		ui->pushButton_modbus_OFF_1->click();
	}

	/* Disable EtherCAT during blind RCM */
	if (config->get_support_ethercat_mode())
	{
		ui->pushButton_etherCAT_OFF_1->click();
	}

	/* Disable logging during blind RCM */
	if (config->get_support_logging())
	{
		ui->pushButton_data_logging_OFF_1->click();
	}

	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_RS232_BLIND);
}


/* Enable 'Live' TCP RCM. */
void GUI_1channel::on_pushButton_remote_command_TCP_1_clicked()
{
	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_TCP_LIVE);
}

/* Enable 'Blind' TCP RCM. */
void GUI_1channel::on_pushButton_remote_command_TCP_blind_1_clicked()
{
	/* Disable Modbus during blind RCM */
	if (config->get_support_modbus_mode())
	{
		ui->pushButton_modbus_OFF_1->click();
	}

	/* Disable EtherCAT during blind RCM */
	if (config->get_support_ethercat_mode())
	{
		ui->pushButton_etherCAT_OFF_1->click();
	}

	/* Disable logging during blind RCM */
	if (config->get_support_logging())
	{
		ui->pushButton_data_logging_OFF_1->click();
	}

	emit signal_remote_command_mode_set(RCM_MODE::RCM_MODE_TCP_BLIND);
}

void GUI_1channel::RCM_controls_enable(bool enable)
{
	for (int i = 0; i < menu_buttonGroup.buttons().count(); i++)
	{
		menu_buttonGroup.buttons().at(i)->setEnabled(enable);
	}

	if (coherency_supported == true)
	{
		for (int i = 0; i < clock_source_buttonGroup.buttons().count(); i++)
		{
			clock_source_buttonGroup.buttons().at(i)->setEnabled(enable);
		}

		ui->label_clock_source_1->setEnabled(enable);
	}

	if (config->get_support_PWM_triggering() == true)
	{
		ui->pushButton_PWM_triggering_free_running_1->setEnabled(enable);
		ui->pushButton_PWM_triggering_master_1->setEnabled(enable);
		ui->pushButton_PWM_triggering_slave_1->setEnabled(enable);
		ui->PWM_triggering_label->setEnabled(enable);
	}

	if (config->get_support_power_control_modes() == true)
	{
		ui->pushButton_power_control_normal_1->setEnabled(enable);
		ui->pushButton_power_control_feedforward_1->setEnabled(enable && config->get_support_feedforward_control());
		ui->pushButton_power_control_analog_1->setEnabled(enable && config->get_support_AIS_mode());
		ui->pushButton_power_control_PID_1->setEnabled(enable && config->get_support_PID_control());
		ui->label_power_control_mode_1->setEnabled(enable);

		if (config->get_support_amplifier_mode() >= 1)
		{
			configure_availability_amplifier_mode();
		}
	}

	if (config->get_support_external_triggering_mode() == true)
	{
		configure_availability_external_triggering();
	}

	if (config->get_support_logging() == true)
	{
		ui->pushButton_data_logging_ON_1->setEnabled(enable);
		ui->pushButton_data_logging_OFF_1->setEnabled(enable);
		ui->label_data_logging_1->setEnabled(enable);
	}

	if (config->get_support_modbus_mode() == true)
	{
		ui->pushButton_modbus_OFF_1->setEnabled(enable);
		ui->pushButton_modbus_TCP_slave_1->setEnabled(enable);
		ui->label_modbus_mode_1->setEnabled(enable);
	}

	if (config->get_support_ethercat_mode() == true)
	{
		ui->pushButton_etherCAT_OFF_1->setEnabled(enable);
		ui->pushButton_etherCAT_ON_1->setEnabled(enable);
		ui->label_etherCAT_mode_1->setEnabled(enable);
	}
}


/**********************************************************************************************************************************************************************************
 * MODBUS
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_pushButton_modbus_OFF_1_clicked()
{
	emit signal_set_modbus_enable(false);
	configure_auto_error_clearing();
}

void GUI_1channel::on_pushButton_modbus_TCP_slave_1_clicked()
{
	emit signal_set_modbus_enable(true);
	configure_auto_error_clearing();
}

void GUI_1channel::handler_modbus_connected_enable(bool enable, QString address, int port, QString error)
{
	if (enable)
	{
		ui->pushButton_modbus_TCP_slave_1->setChecked(true);
		ui->settings_plainTextEdit->appendPlainText(QString(">\tModbus Server is listening for connections at:\n\tIP:\t\t%1\n\tPort:\t%2").arg(address).arg(port));
	}
	else
	{
		ui->pushButton_modbus_OFF_1->setChecked(true);
		if (error.isEmpty())
		{
			ui->settings_plainTextEdit->appendPlainText(QString(">\tModbus Server has stopped listening for connections."));
		}
		else
		{
			ui->settings_plainTextEdit->appendPlainText(">\tModbus Error:\n\t" + error);
		}
	}
}

/* Set the start condition for Modbus */
void GUI_1channel::handler_modbus_mode()
{
	if (config->get_support_modbus_mode() == true)
	{
		if (config->get_modbus_enabled() == true)
		{
			ui->pushButton_modbus_TCP_slave_1->click();
		}
		else
		{
			ui->pushButton_modbus_OFF_1->click();
		}
	}
}


/**********************************************************************************************************************************************************************************
 * EtherCAT
 * *******************************************************************************************************************************************************************************/

void GUI_1channel::on_pushButton_etherCAT_OFF_1_clicked()
{
	emit signal_set_ethercat_enable(false);
	configure_auto_error_clearing();
}

void GUI_1channel::on_pushButton_etherCAT_ON_1_clicked()
{
	emit signal_set_ethercat_enable(true);
	configure_auto_error_clearing();
}

void GUI_1channel::handler_ethercat_mode()
{
	if (config->get_support_ethercat_mode() == true)
	{
		if (config->get_ethercat_enabled() == true)
		{
			ui->pushButton_etherCAT_ON_1->click();
		}
		else
		{
			ui->pushButton_etherCAT_OFF_1->click();
		}
	}
}


/**********************************************************************************************************************************************************************************
 * CHAIN MODE
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_pushButton_clock_source_standalone_1_clicked()
{
	emit signal_setClock_source(1,0);
}

void GUI_1channel::on_pushButton_clock_source_master_1_clicked()
{
	emit signal_setClock_source(1,1);
}

void GUI_1channel::on_pushButton_clock_source_slave_end_1_clicked()		//doesn't really get used, should probably be removed altogether.
{
	emit signal_setClock_source(1,2);
}

void GUI_1channel::on_pushButton_clock_source_slave_inline_1_clicked()
{
	emit signal_setClock_source(1,3);
}

void GUI_1channel::on_pushButton_clock_source_AUX_master_1_clicked()
{
	emit signal_setClock_source(1,4);
}

void GUI_1channel::on_pushButton_clock_source_AUX_slave_end_1_clicked()
{
	emit signal_setClock_source(1,5);
}

/**********************************************************************************************************************************************************************************
 * Power Control Modes
 * *******************************************************************************************************************************************************************************/
/* Set Normal Power Control Mode: Autogain Enabled */
void GUI_1channel::on_pushButton_power_control_normal_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_NORMAL);
}

/* Set Analog Power Control Mode. Voltage controlled $GCS or $MCS. The Modulator type is automatically chosen depending on the enable state of ALL. */
void GUI_1channel::on_pushButton_power_control_analog_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_ANALOG_INPUT);
}

/* Set Feed Forward Power Control Mode: Autogain Disabled. */
void GUI_1channel::on_pushButton_power_control_feedforward_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_FEED_FWD);
}

/* Set Amplifier Power Control Mode: The small control signal is provided externally, while the system serves only as an amplifier. Exact functionality is hardware dependent. */
void GUI_1channel::on_pushButton_power_control_amplifier_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_AMPLIFIER);
}

/* Set PID Power Control Mode: The Power output is regulated based on a feedback loop with an external sensor. */
void GUI_1channel::on_pushButton_power_control_PID_1_clicked()
{
	emit signal_set_power_control_mode(1, POWER_CONTROL_PID);
}

/**********************************************************************************************************************************************************************************
 * External Triggering Mode
 * *******************************************************************************************************************************************************************************/
/* External triggering mode disabled; Enable PWM controls and set duty cycle to 100% */
void GUI_1channel::on_pushButton_external_triggering_OFF_1_clicked()
{
	emit signal_setExternal_triggering_enable(1, false);
}

/* External triggering mode enabled; Disable PWM controls and set duty cycle to 0% so that we can do external triggering */
void GUI_1channel::on_pushButton_external_triggering_ON_1_clicked()
{
	emit signal_setExternal_triggering_enable(1, true);
}

/**********************************************************************************************************************************************************************************
 * PWM TRIGGERING
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_pushButton_PWM_triggering_free_running_1_clicked()
{
	signal_setPWM_triggering_mode(1, 1);
}

void GUI_1channel::on_pushButton_PWM_triggering_master_1_clicked()
{
	signal_setPWM_triggering_mode(1, 2);
}

void GUI_1channel::on_pushButton_PWM_triggering_slave_1_clicked()
{
	signal_setPWM_triggering_mode(1, 3);
}

/**********************************************************************************************************************************************************************************
 * S11 SWEEP
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_pushButton_SWP_execute_1_clicked()
{
	SWP_enable_GUI(false);
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Sweep_graph_page));

	ui->SWP_plot->clearGraphs();
	ui->SWP_plot->replot();
	emit signal_execute_SWP(1);
}

void GUI_1channel::on_pushButton_SWP_execute_2_clicked()
{
	SWP_enable_GUI(false);
	ui->SWP_plot->clearGraphs();
	ui->SWP_plot->replot();
	emit signal_execute_SWP(1);
}

void GUI_1channel::on_pushButton_SWP_unit_clicked()
{
	set_power_reading_units_watt(ui->pushButton_SWP_unit->isChecked());
	SWP_draw_plot(RF_System::Channels->at(0)->SWP_Raw_Data());
}

void GUI_1channel::on_pushButton_SWP_back_clicked()
{
	ui->pushButton_SWP_execute_1->setChecked(false);
	ui->stackedWidget->setCurrentIndex(ui->stackedWidget->indexOf(ui->sweep_controls_page));
	ui->stackedWidget_2->setCurrentIndex(ui->stackedWidget_2->indexOf(ui->page_logo));
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
}

/* Sets the UI state during and after sweeps */
void GUI_1channel::SWP_enable_GUI(bool enable)
{
	enable == true ? ui->pushButton_SWP_execute_1->setText("Sweep") : ui->pushButton_SWP_execute_1->setText("Wait...");
	enable == true ? ui->pushButton_SWP_execute_2->setText("Sweep") : ui->pushButton_SWP_execute_2->setText("Please\nWait...");
	ui->pushButton_SWP_execute_1->setChecked(!enable);
	ui->pushButton_SWP_execute_2->setChecked(!enable);
	ui->pushButton_SWP_back->setEnabled(enable);
	ui->pushButton_SWP_unit->setEnabled(enable);

	ui->pushButton_SWP_set_best_frequency_1->setEnabled(enable);
	ui->pushButton_SWP_set_best_frequency_1->setText("Use Frequency");

	if (enable == false)
	{
		ui->label_SWP_best_match_2->setText("-");
		ui->label_SWP_best_frequency_2->setText("-");
	}

	ui->label_SWP_best_match_1->setEnabled(enable);
	ui->label_SWP_best_match_2->setEnabled(enable);
	ui->label_SWP_best_match_3->setEnabled(enable);
	ui->label_SWP_best_frequency_1->setEnabled(enable);
	ui->label_SWP_best_frequency_2->setEnabled(enable);
	ui->label_SWP_best_frequency_3->setEnabled(enable);

	for (int i = 0; i < menu_buttonGroup.buttons().count(); i++)
	{
		menu_buttonGroup.buttons().at(i)->setEnabled(enable);
	}
}

void GUI_1channel::on_pushButton_SWP_set_best_frequency_1_clicked()
{
	emit signal_setFrequency(1, best_S11_match_frequency);
	emit signal_setDLL_start_frequency(1, best_S11_match_frequency);
	ui->pushButton_SWP_set_best_frequency_1->setText("Done!");
	ui->pushButton_SWP_set_best_frequency_1->setEnabled(false);

	//
	//TODO:
	//Perhaps a little notification / warning if the DLL start freq is outside the DLL limits...
	//
}

/**********************************************************************************************************************************************************************************
 * POWER MENU
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::on_pushButton_exit_program_1_clicked()
{
	power_menu_action = 3;
	ui->power_menu_confirm_label->setText("Are you sure you want to exit the program?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel::on_pushButton_reset_SGx_1_clicked()
{
	power_menu_action = 5;
	ui->power_menu_confirm_label->setText("Are you sure you want to reset the SGx board?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel::on_pushButton_restart_program_1_clicked()
{
	power_menu_action = 7;
	ui->power_menu_confirm_label->setText("Are you sure you want to restart the program?");
	ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->power_menu_confirm_page));
}

void GUI_1channel::on_pushButton_clear_errors_1_clicked()
{
	emit signal_execute_error_clear(1);
}

void GUI_1channel::on_Exit_Yes_button_clicked()
{
	QProcess process;

	switch(power_menu_action)
	{
		case 3:

		#if defined(Q_OS_LINUX)
			RF_system_constructor->handler_LED_mode_set(0, LED_MODE::LED_MODE_OFF);		//If RPi HAT is used, disable the LED.
		#endif

			this->close();
			break;

//		case 4:
//			ui->stackedWidget_5->setCurrentIndex(ui->stackedWidget_5->indexOf(ui->page_overview));
//			if (config->get_reset_on_startup() == true)
//			{
//				emit signal_execute_reset_SGx(1);
//			}
//			break;

		case 5:

			emit signal_execute_reset_SGx(1);
			ui->menu_home_Button->click();	//Always start from the Home menu.
			break;

//		case 6:
//			emit signal_execute_reset_protection(1);
//			ui->stackedWidget_3->setCurrentIndex(last_page_index);
//			break;

		case 7:


		#if defined(Q_OS_LINUX)
			RF_system_constructor->handler_LED_mode_set(0, LED_MODE::LED_MODE_OFF);		//If RPi HAT is used, disable the LED.
		#endif

			handler_execute_restart_program();
			break;

		default:
			break;
	}
}


void GUI_1channel::on_Exit_No_button_clicked()
{
	ui->stackedWidget_3->setCurrentIndex(last_page_index);
}

void GUI_1channel::handler_execute_restart_program()
{
	QProcess process;
	process.setProgram(QCoreApplication::applicationDirPath() + "/" + QCoreApplication::applicationName());
	qDebug() << "GUI: Restarting -" << process.program();
	process.startDetached();
	this->close();
}

/**********************************************************************************************************************************************************************************
 * Channel reading handlers
 * *******************************************************************************************************************************************************************************/


void GUI_1channel::handler_firmware_version_get(int intrasys_num, int channel_num, QString version)
{
	(void) intrasys_num, (void)channel_num;

	SG_firmware_version = version;
	update_listWidgetItem_system();

	if (version != "")
	{
		firmware_version_check(version);
		ui->firmware_version_label_2->setText(version);
	}
}

void GUI_1channel::handler_board_identity_get(int intrasys_num, QString manufacturer, QString model, QString serialnumber)
{
	(void) intrasys_num;

	SG_manufacturer = manufacturer;
	SG_board_model = model;
	SG_serial_number = serialnumber;
	update_listWidgetItem_system();

	if(!serialnumber.isEmpty())
	{
		ui->SG_board_identity_label_2->setText(serialnumber);
	}
	else
	{
		ui->SG_board_identity_label_1->setVisible(false);
		ui->SG_board_identity_label_2->setVisible(false);
	}
}

void GUI_1channel::handler_hardware_identity_get(int intrasys_num, int identity_number, QString identity_string)
{
	(void) intrasys_num;

	hardware_identity_string = identity_string;
	hardware_identity_number = identity_number;
	update_listWidgetItem_system();
}

void GUI_1channel::handler_PA_type_get(int intrasys_num, int PA_type)
{
	(void) intrasys_num;

	this->PA_type = PA_type;
	update_listWidgetItem_system();
}


void GUI_1channel::handler_error_get(int intrasys_num, quint64 error, QStringList error_messages)
{
	(void) intrasys_num;

	if (error > 0)
	{
		//display in error label.
		ui->label_error_2->setText(QString("0x%1").arg(error,0,16));

		QString message_string = "";
		for (int i = 0; i < error_messages.count(); i++)
		{
			if (error_messages.at(i) != "")
			{
				message_string += error_messages.at(i);
			}
			if (i != error_messages.count() - 1)
			{
				message_string += "\n";
			}
		}

		show_error(message_string);
	}
	else
	{
		ui->label_error_2->setText(error_messages.at(0));
		close_error();
	}

	//datalogging
	datalogger->setData(9, QString::number(error,16));
}

void GUI_1channel::handler_reset_detected(int intrasys_num, int channel_number)
{
	(void) intrasys_num, (void) channel_number;

	show_notification("A reset has occurred");
	ui->settings_plainTextEdit->appendPlainText(">\tSystem has been reset to defaults.");

	/* Configure Power Supply start state */
	PSU_reset();
}

void GUI_1channel::handler_temperature_PA_get(int intrasys_num, double temperature_PA)
{
	(void) intrasys_num;

	ui->label_temperature_2->setText(QString::number(temperature_PA,'f',1));

	//datalogging
	datalogger->setData(8, QString::number(temperature_PA));
}

void GUI_1channel::handler_temperature_SG_get(int intrasys_num, double temperature_SG)
{
	(void) intrasys_num;

	datalogger->setData(12, QString::number(temperature_SG));
}

void GUI_1channel::handler_power_control_mode_get(int intrasys_num, int mode)
{
	int showPWRS = config->get_feedforward_PWRS_controls_enabled();
	int feedFWD_mode = config->get_feedforward_mode();

	close_notification();

	switch(mode)
	{
		case POWER_CONTROL_NORMAL:
			show_amplifier_mode_controls(intrasys_num, false);
			show_autogain_controls(intrasys_num, true, true);
			show_sweep_power_controls(intrasys_num, true, true);
			show_GCS_controls(intrasys_num, false, false);
			show_MCS_controls(intrasys_num, false, false);
			ui->pushButton_power_control_normal_1->setChecked(true);
			break;

		case POWER_CONTROL_ANALOG_INPUT:
			show_autogain_controls(intrasys_num, false, false);
			show_sweep_power_controls(intrasys_num, false, true);
			show_amplifier_mode_controls(intrasys_num, false);
			show_GCS_controls(intrasys_num, true, true);
			show_MCS_controls(intrasys_num, true, true);
			ui->pushButton_power_control_analog_1->setChecked(true);
			break;

		case POWER_CONTROL_FEED_FWD:
			if (showPWRS >= 0)
			{
				show_autogain_controls(intrasys_num, (bool)showPWRS, (bool)showPWRS);
				show_sweep_power_controls(intrasys_num, (bool)showPWRS, true);
			}
			else
			{
				show_autogain_controls(intrasys_num, false, false);
				show_sweep_power_controls(intrasys_num, false, true);
			}

			show_amplifier_mode_controls(intrasys_num, false);

			if (feedFWD_mode == 0)
			{
				show_GCS_controls(intrasys_num, false, true);
				show_MCS_controls(intrasys_num, false, true);
			}
			else if (feedFWD_mode >= 1)
			{
				show_GCS_controls(intrasys_num, (bool)(feedFWD_mode - 1), true);
				show_MCS_controls(intrasys_num, (bool)!(feedFWD_mode - 1), true);
			}
			else
			{
				show_GCS_controls(intrasys_num, true, true);
				show_MCS_controls(intrasys_num, true, true);
			}

			ui->pushButton_power_control_feedforward_1->setChecked(true);
			break;

		case POWER_CONTROL_AMPLIFIER:
			show_autogain_controls(intrasys_num, false, false);
			show_sweep_power_controls(intrasys_num, false, true);
			show_amplifier_mode_controls(intrasys_num, true);

			/* Show the appropriate controls */
			switch (config->get_AM_controls_enabled())
			{
				case 0:		//No MCS, No GCS
					show_MCS_controls(intrasys_num, false, true);
					show_GCS_controls(intrasys_num, false, true);
					break;

				case 1:		//yes MCS, No GCS
					show_MCS_controls(intrasys_num, true, true);
					show_GCS_controls(intrasys_num, false, true);
					break;

				case 2:		//no MCS, yes GCS
					show_MCS_controls(intrasys_num, false, true);
					show_GCS_controls(intrasys_num, true, true);
					break;

				case 3:		//yes MCS, yes GCS
					show_MCS_controls(intrasys_num, true, true);
					show_GCS_controls(intrasys_num, true, true);
					break;

				default:	//No MCS, No GCS	- same as 0;
					show_MCS_controls(intrasys_num, false, true);
					show_GCS_controls(intrasys_num, false, true);
					break;
			}

			ui->pushButton_power_control_amplifier_1->setChecked(true);

			break;

		case POWER_CONTROL_PID:
			show_amplifier_mode_controls(intrasys_num, false);
			show_autogain_controls(intrasys_num, false, false);
			show_sweep_power_controls(intrasys_num, false, false);
			show_GCS_controls(intrasys_num, false, false);
			show_MCS_controls(intrasys_num, false, false);
			ui->pushButton_power_control_PID_1->setChecked(true);
			break;

		case POWER_CONTROL_UNKNOWN:
			ui->pushButton_power_control_normal_1->setChecked(false);
			ui->pushButton_power_control_analog_1->setChecked(false);
			ui->pushButton_power_control_feedforward_1->setChecked(false);
			ui->pushButton_power_control_PID_1->setChecked(false);
			show_autogain_controls(intrasys_num, false, false);
			show_sweep_power_controls(intrasys_num, false, false);
			show_GCS_controls(intrasys_num, false, false);
			show_MCS_controls(intrasys_num, false, false);

			ui->settings_plainTextEdit->appendPlainText(">\tUnknown Power Control Mode Configured!");
			show_notification("Unknown Power Control Mode Configured!", 16000);
			break;

		default:
			break;
	}
}

void GUI_1channel::handler_RF_enable_get(int intrasys_num, bool RF_enabled)
{
	(void) intrasys_num;

	if (RF_enabled == true)
	{
		ui->pushButton_RF_enable_1->setText("RF is ON");

		//datalogging
		datalogger->setData(0, "true");
	}
	else
	{
		ui->pushButton_RF_enable_1->setText("RF is OFF");

		//datalogging
		datalogger->setData(0, "false");
	}

	ui->pushButton_RF_enable_1->setChecked(RF_enabled);
}

void GUI_1channel::handler_PA_power_readings(int intrasys_num, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection)
{
	(void) intrasys_num;

	double PA_power_minimum_dbm = 30;

	if (ui->pushButton_unit_power_forward_1->isChecked())
	{
		ui->label_power_forward->setText(QString::number(PA_power_fwd_watt,'f',1));
	}
	else
	{
		ui->label_power_forward->setText(QString::number(PA_power_fwd_dbm,'f',1));
	}

	if (ui->pushButton_unit_power_reflected_1->isChecked())
	{
		ui->label_power_reflected->setText(QString::number(PA_power_rfl_watt,'f',1));
	}
	else
	{
		ui->label_power_reflected->setText(QString::number(PA_power_rfl_dbm,'f',1));
	}

	if (PA_power_fwd_dbm >= PA_power_minimum_dbm)
	{
		if (ui->pushButton_unit_S11_1->isChecked())
		{
			ui->label_s11_value->setText(QString::number(PA_s11_reflection,'f',1));
		}
		else
		{
			ui->label_s11_value->setText(QString::number(PA_s11,'f',1));
		}
	}
	else
	{
		ui->label_s11_value->setText("-");
	}

	//datalogging
	datalogger->setData(3, QString::number(PA_power_fwd_watt,'f',1));
	datalogger->setData(4, QString::number(PA_power_rfl_watt,'f',1));
	datalogger->setData(5, QString::number(PA_s11_reflection,'f',1));
}

void GUI_1channel::handler_PA_power_readings_NChannel(int intrasys_num, int subchannel, double PA_power_fwd_dbm, double PA_power_rfl_dbm, double PA_s11, double PA_power_fwd_watt, double PA_power_rfl_watt, double PA_s11_reflection)
{
	(void) intrasys_num, (void) PA_s11, (void)PA_s11_reflection;

	if (ui->pushButton_unit_power_forward_1->isChecked())
	{
		ui->label_splitter_FWD_CH1_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH2_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH3_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH4_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH5_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH6_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH7_1->setText("FWD (W):");
		ui->label_splitter_FWD_CH8_1->setText("FWD (W):");

		switch (subchannel)
		{
			case 1:
				ui->label_splitter_FWD_CH1_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 2:
				ui->label_splitter_FWD_CH2_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 3:
				ui->label_splitter_FWD_CH3_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 4:
				ui->label_splitter_FWD_CH4_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 5:
				ui->label_splitter_FWD_CH5_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 6:
				ui->label_splitter_FWD_CH6_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 7:
				ui->label_splitter_FWD_CH7_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;
			case 8:
				ui->label_splitter_FWD_CH8_2->setText(QString::number(PA_power_fwd_watt,'f',1));
				break;

			default:
				break;
		}
	}
	else
	{
		ui->label_splitter_FWD_CH1_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH2_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH3_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH4_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH5_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH6_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH7_1->setText("FWD (dBm):");
		ui->label_splitter_FWD_CH8_1->setText("FWD (dBm):");

		switch (subchannel)
		{
			case 1:
				ui->label_splitter_FWD_CH1_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 2:
				ui->label_splitter_FWD_CH2_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 3:
				ui->label_splitter_FWD_CH3_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 4:
				ui->label_splitter_FWD_CH4_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 5:
				ui->label_splitter_FWD_CH5_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 6:
				ui->label_splitter_FWD_CH6_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 7:
				ui->label_splitter_FWD_CH7_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;
			case 8:
				ui->label_splitter_FWD_CH8_2->setText(QString::number(PA_power_fwd_dbm,'f',1));
				break;

			default:
				break;
		}
	}

	if (ui->pushButton_unit_power_reflected_1->isChecked())
	{
		ui->label_splitter_RFL_CH1_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH2_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH3_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH4_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH5_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH6_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH7_1->setText("RFL (W):");
		ui->label_splitter_RFL_CH8_1->setText("RFL (W):");

		switch (subchannel)
		{
			case 1:
				ui->label_splitter_RFL_CH1_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 2:
				ui->label_splitter_RFL_CH2_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 3:
				ui->label_splitter_RFL_CH3_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 4:
				ui->label_splitter_RFL_CH4_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 5:
				ui->label_splitter_RFL_CH5_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 6:
				ui->label_splitter_RFL_CH6_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 7:
				ui->label_splitter_RFL_CH7_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;
			case 8:
				ui->label_splitter_RFL_CH8_2->setText(QString::number(PA_power_rfl_watt,'f',1));
				break;

			default:
				break;
		}
	}
	else
	{
		ui->label_splitter_RFL_CH1_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH2_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH3_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH4_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH5_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH6_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH7_1->setText("RFL (dBm):");
		ui->label_splitter_RFL_CH8_1->setText("RFL (dBm):");

		switch (subchannel)
		{
			case 1:
				ui->label_splitter_RFL_CH1_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 2:
				ui->label_splitter_RFL_CH2_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 3:
				ui->label_splitter_RFL_CH3_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 4:
				ui->label_splitter_RFL_CH4_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 5:
				ui->label_splitter_RFL_CH5_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 6:
				ui->label_splitter_RFL_CH6_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 7:
				ui->label_splitter_RFL_CH7_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;
			case 8:
				ui->label_splitter_RFL_CH8_2->setText(QString::number(PA_power_rfl_dbm,'f',1));
				break;

			default:
				break;
		}
	}
}

void GUI_1channel::handler_phase_NChannel_get(int intrasys_num, int subchannel, double phase_degrees)
{
	(void) intrasys_num;

	switch (subchannel)
	{
		case 1:
			ui->pushButton_splitter_phase_CH1->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 2:
			ui->pushButton_splitter_phase_CH2->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 3:
			ui->pushButton_splitter_phase_CH3->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 4:
			ui->pushButton_splitter_phase_CH4->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 5:
			ui->pushButton_splitter_phase_CH5->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 6:
			ui->pushButton_splitter_phase_CH6->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 7:
			ui->pushButton_splitter_phase_CH7->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;
		case 8:
			ui->pushButton_splitter_phase_CH8->setText(zeroChopper(QString::number(phase_degrees,'f',1)));
			break;


		default:
			break;
	}
}

void GUI_1channel::handler_attenuation_NChannel_get(int intrasys_num, int subchannel, double attenuation)
{
	(void) intrasys_num;

	switch (subchannel)
	{
		case 1:
			ui->pushButton_splitter_attenuation_CH1->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 2:
			ui->pushButton_splitter_attenuation_CH2->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 3:
			ui->pushButton_splitter_attenuation_CH3->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 4:
			ui->pushButton_splitter_attenuation_CH4->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 5:
			ui->pushButton_splitter_attenuation_CH5->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 6:
			ui->pushButton_splitter_attenuation_CH6->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 7:
			ui->pushButton_splitter_attenuation_CH7->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;
		case 8:
			ui->pushButton_splitter_attenuation_CH8->setText(zeroChopper(QString::number(attenuation,'f',2)));
			break;

		default:
			break;
	}
}

void GUI_1channel::handler_magnitude_NChannel_get(int intrasys_num, int subchannel, double magnitude)
{
	(void) intrasys_num;

	switch (subchannel)
	{
		case 1:
			ui->pushButton_splitter_magnitude_CH1->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 2:
			ui->pushButton_splitter_magnitude_CH2->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 3:
			ui->pushButton_splitter_magnitude_CH3->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 4:
			ui->pushButton_splitter_magnitude_CH4->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 5:
			ui->pushButton_splitter_magnitude_CH5->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 6:
			ui->pushButton_splitter_magnitude_CH6->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 7:
			ui->pushButton_splitter_magnitude_CH7->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;
		case 8:
			ui->pushButton_splitter_magnitude_CH8->setText(zeroChopper(QString::number(magnitude,'f',1)));
			break;

		default:
			break;
	}
}


void GUI_1channel::handler_power_get(int intrasys_num, double power_dbm, double power_watt)
{
	(void) intrasys_num;

	ui->pushButton_power_dbm_1->setText(zeroChopper(QString::number(power_dbm, 'f', 2)));
	ui->pushButton_power_watt_1->setText(zeroChopper(QString::number(power_watt, 'f', 2)));
	ui->pushButton_power_dbm_2->setText(zeroChopper(QString::number(power_dbm, 'f', 2)));
	ui->pushButton_power_watt_2->setText(zeroChopper(QString::number(power_watt, 'f', 2)));

	//datalogging
	datalogger->setData(2, QString::number(power_watt));
}

void GUI_1channel::handler_VGA_attenuation_get(int intrasys_num, double attenuation)
{
	(void) intrasys_num;

	ui->pushButton_VGA_attenuation_1->setText(zeroChopper(QString::number(attenuation, 'f', 2)));
	datalogger->setData(10, QString::number(attenuation));
}

void GUI_1channel::handler_IQMod_magnitude_get(int intrasys_num, double magnitude)
{
	(void) intrasys_num;

	ui->pushButton_IQMod_magnitude_1->setText(zeroChopper(QString::number(magnitude, 'f', 1)));
	datalogger->setData(11, QString::number(magnitude));
}

void GUI_1channel::handler_clock_source_get(int intrasys_num, int mode)
{
	(void) intrasys_num;

	/* protection against out of index crash if a mode is configured that does not have a corresponding button. */
	bool valid_clock_source_mode = (mode < clock_source_buttonGroup.buttons().count());

	if (valid_clock_source_mode == true)
	{
		if (clock_source_buttonGroup.buttons().at(mode)->isChecked() == false)
		{
			clock_source_buttonGroup.buttons().at(mode)->setChecked(true);
		}
	}
	else
	{
		qDebug() << "Unknown clock source mode: " << mode;
		return;
	}
}

void GUI_1channel::handler_PWM_triggering_mode_get(int intrasys_num, int mode)
{
	(void) intrasys_num;

	PWM_triggering_buttonGroup.buttons().at(mode-1)->setChecked(true);
}

//CW
void GUI_1channel::handler_frequency_get(int intrasys_num, double frequency_mhz)
{
	(void) intrasys_num;

	ui->pushButton_frequency_1->setText(zeroChopper(QString::number(frequency_mhz, 'f', 2)));

	/* DLL lock frequency indication with flexible precision, to avoid annoying jumpiness in the UI */
	QString string1 = zeroChopper(QString::number(frequency_mhz,'f',3));
	QString string2 = ui->pushButton_DLL_step_frequency_2->text();
	int lock_precision = 0;

	if (string1.contains("."))
	{
		lock_precision = string1.count() - string1.indexOf(".") - 1;
	}
	if (string2.contains("."))
	{
		int temp_precision = string2.count() - string2.indexOf(".") - 1;
		if (temp_precision > lock_precision)
		{
			lock_precision = temp_precision;
		}
	}

	ui->label_DLL_frequency_lock_2->setText(QString::number(frequency_mhz, 'f', lock_precision));

	//datalogging
	datalogger->setData(1, QString::number(frequency_mhz));
}

void GUI_1channel::handler_offset_db_get(int intrasys_num, double offset_db)
{
	(void) intrasys_num;

	ui->pushButton_power_offset_dB_1->setText(zeroChopper(QString::number(offset_db, 'f', 2)));
}

void GUI_1channel::handler_phase_get(int intrasys_num, double phase_degrees)
{
	(void) intrasys_num;

	ui->pushButton_phase_1->setText(zeroChopper(QString::number(phase_degrees, 'f', 1 )));
}

void GUI_1channel::handler_CW_enable_get(int intrasys_num, bool enable)
{
	(void) intrasys_num;

	ui->pushButton_CW_enable_1->setChecked(enable);
}

//PWM
void GUI_1channel::handler_PWM_settings_get(int intrasys_num, double frequency_hz, int duty_cycle)
{
	(void) intrasys_num;

	if (duty_cycle <= 0)							//PWM disabled <> External Triggering Mode enabled.
	{
		ui->pushButton_PWM_enable_1->setChecked(false);
		show_PWM_settings(false);
	}
	else if (duty_cycle >= 1 && duty_cycle <= 99)	//PWM enabled <> External Triggering Mode disabled.
	{
		ui->pushButton_PWM_enable_1->setChecked(true);
		show_PWM_settings(true);
	}
	else											//PWM disabled <> External Triggering Mode disabled.
	{
		ui->pushButton_PWM_enable_1->setChecked(false);
		show_PWM_settings(false);
	}

	ui->pushButton_PWM_duty_cycle_1->setText(zeroChopper(QString::number(duty_cycle,'f',1)));
	ui->pushButton_PWM_frequency_1->setText(zeroChopper(QString::number(frequency_hz, 'f', 2)));

	//datalogging
	datalogger->setData(6, QString::number(frequency_hz));
	datalogger->setData(7, QString::number(duty_cycle));
}

void GUI_1channel::handler_external_triggering_enable_get(int intrasys_num, bool enable)
{
	(void) intrasys_num;

	if (enable == true)
	{
		ui->pushButton_external_triggering_ON_1->setChecked(true);
	}
	else
	{
		ui->pushButton_external_triggering_OFF_1->setChecked(true);
	}
}

//DLL
void GUI_1channel::handler_DLL_enable_get(int intrasys_num, bool enable)
{
	(void) intrasys_num;

	ui->pushButton_DLL_enable_1->setChecked(enable);
	show_DLL_settings(enable);
}

void GUI_1channel::handler_DLL_settings_get(int intrasys_num, double frequency_lower, double frequency_upper, double frequency_start, double frequency_step, double threshold, double main_delay)
{
	(void) intrasys_num;

	ui->pushButton_DLL_frequency_limit_lower_1->setText(zeroChopper(QString::number(frequency_lower, 'f', 2)));
	ui->pushButton_DLL_frequency_limit_upper_1->setText(zeroChopper(QString::number(frequency_upper, 'f', 2)));
	ui->pushButton_DLL_start_frequency_1->setText(zeroChopper(QString::number(frequency_start, 'f', 2)));
	ui->pushButton_DLL_step_frequency_1->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->pushButton_DLL_step_frequency_2->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->pushButton_DLL_threshold_1->setText(zeroChopper(QString::number(threshold, 'f', 2)));
	ui->pushButton_DLL_threshold_2->setText(zeroChopper(QString::number(threshold, 'f', 2)));
	ui->pushButton_DLL_main_delay_1->setText(zeroChopper(QString::number(main_delay, 'f', 2)));
}

//ALL
void GUI_1channel::handler_ALL_compatible_get(int intrasys_num, bool compatibility)
{
	(void) intrasys_num;

	//
	// TODO:
	// While there doesn't appear to be a conflict at this time,
	// This function 'handler_ALL_compatible_get()' and 'handler_availability_ALL()'
	// both serve a very similar purpose and should be combined into one.
	//

	if (compatibility == true)
	{
		show_ALL_settings(config->get_menu_ALL_enabled());
		ui->menu_ALL_Button->setVisible(config->get_menu_ALL_enabled());					//enable or disable the ALL menu button
		ui->menu_ALL_Button->setEnabled(config->get_menu_ALL_enabled());
		ui->pushButton_ALL_enable_1->setVisible(config->get_controls_ALL_enabled());		//enable or disable the ALL enable button
		ui->pushButton_ALL_enable_1->setEnabled(config->get_controls_ALL_enabled());
	}
	else
	{
		//
		// TODO:
		// visibility disabled settings can still be bypassed by start-up state / RCM on SG4A's
		//
		show_ALL_settings(false);
		ui->menu_ALL_Button->setVisible(false);				//enable or disable the ALL menu button
		ui->menu_ALL_Button->setEnabled(false);
		ui->pushButton_ALL_enable_1->setVisible(false);		//enable or disable the ALL enable button
		ui->pushButton_ALL_enable_1->setEnabled(false);
	}
}

void GUI_1channel::handler_ALL_enable_get(int intrasys_num, bool enable)
{
	(void) intrasys_num;

	ui->pushButton_ALL_enable_1->setChecked(enable);
	show_ALL_settings(enable);

	//
	// TODO:
	// ALL Compatibility missing...
	//

	ui->pushButton_DVGA_attenuation_forward_1->setEnabled(enable);
	ui->pushButton_DVGA_amplifier_forward_1->setEnabled(enable);
	ui->label_DVGA_attenuation_forward_1->setEnabled(enable);
	ui->label_DVGA_amplifier_forward_1->setEnabled(enable);
	ui->pushButton_DVGA_attenuation_reflected_1->setEnabled(enable);
	ui->pushButton_DVGA_amplifier_reflected_1->setEnabled(enable);
	ui->label_DVGA_attenuation_reflected_1->setEnabled(enable);
	ui->label_DVGA_amplifier_reflected_1->setEnabled(enable);
}

void GUI_1channel::handler_ALL_settings_get(int intrasys_num, double frequency_lower_voltage, double frequency_upper_voltage, double threshold_voltage)
{
	(void) intrasys_num;

	ui->pushButton_ALL_frequency_limit_lower_1->setText(zeroChopper(QString::number(frequency_lower_voltage, 'f', 2)));
	ui->pushButton_ALL_frequency_limit_upper_1->setText(zeroChopper(QString::number(frequency_upper_voltage, 'f', 2)));
	ui->pushButton_ALL_threshold_1->setText(zeroChopper(QString::number(threshold_voltage, 'f', 2)));
	ui->pushButton_ALL_threshold_2->setText(zeroChopper(QString::number(threshold_voltage, 'f', 2)));
}

//DVGA's
void GUI_1channel::handler_DVGA_compatible_get(int intrasys_num, bool compatibility)
{
	(void) intrasys_num;

	ui->label_DVGA_attenuation_forward_1->setVisible(compatibility);
	ui->label_DVGA_attenuation_reflected_1->setVisible(compatibility);
	ui->label_DVGA_amplifier_forward_1->setVisible(compatibility);
	ui->label_DVGA_amplifier_reflected_1->setVisible(compatibility);
	ui->pushButton_DVGA_attenuation_forward_1->setVisible(compatibility);
	ui->pushButton_DVGA_attenuation_reflected_1->setVisible(compatibility);
	ui->pushButton_DVGA_amplifier_forward_1->setVisible(compatibility);
	ui->pushButton_DVGA_amplifier_reflected_1->setVisible(compatibility);
}

void GUI_1channel::handler_DVGA_forward_settings_get(int intrasys_num, bool amplifier_enable, double attenuation)
{
	(void) intrasys_num;

	ui->pushButton_DVGA_attenuation_forward_1->setText(zeroChopper(QString::number(attenuation, 'f', 2)));
	if(amplifier_enable == true)
	{
		ui->pushButton_DVGA_amplifier_forward_1->setText("ON");
	}
	else
	{
		ui->pushButton_DVGA_amplifier_forward_1->setText("OFF");
	}

	ui->pushButton_DVGA_amplifier_forward_1->setChecked(amplifier_enable);
}

void GUI_1channel::handler_DVGA_reflected_settings_get(int intrasys_num, bool amplifier_enable, double attenuation)
{
	(void) intrasys_num;

	ui->pushButton_DVGA_attenuation_reflected_1->setText(zeroChopper(QString::number(attenuation, 'f', 2)));
	if(amplifier_enable == true)
	{
		ui->pushButton_DVGA_amplifier_reflected_1->setText("ON");
	}
	else
	{
		ui->pushButton_DVGA_amplifier_reflected_1->setText("OFF");
	}

	ui->pushButton_DVGA_amplifier_reflected_1->setChecked(amplifier_enable);
}

//SWP
void GUI_1channel::handler_SWP_settings_get(int intrasys_num, double frequency_start, double frequency_stop, double frequency_step, double power_dbm, double power_watt)
{
	(void) intrasys_num;

	ui->pushButton_SWP_start_frequency_1->setText(zeroChopper(QString::number(frequency_start, 'f', 3)));
	ui->pushButton_SWP_stop_frequency_1->setText(zeroChopper(QString::number(frequency_stop, 'f', 3)));
	ui->pushButton_SWP_step_frequency_1->setText(zeroChopper(QString::number(frequency_step, 'f', 3)));
	ui->pushButton_SWP_power_dbm_1->setText(zeroChopper(QString::number(power_dbm, 'f', 1)));
	ui->pushButton_SWP_power_watt_1->setText(zeroChopper(QString::number(power_watt, 'f', 1)));
}

void GUI_1channel::handler_SWP_measurement_get(int intrasys_num, QString SWP_raw_data)
{
	(void) intrasys_num;

	if(SWP_raw_data.contains("$SWPD,") && SWP_raw_data.contains("OK\r\n"))
	{
		SWP_draw_plot(SWP_raw_data);
	}
	else
	{
		ui->stackedWidget_3->setCurrentIndex(ui->stackedWidget_3->indexOf(ui->Main_page));
		if (SWP_raw_data != "")
		{
			QStringList response = SWP_raw_data.split(",");
			if (response.count() >= 3)
			{
				QString error = response.at(2);
				error.chop(2);
				show_notification("Sweep returned error: " + error);
			}
		}
		else
		{
			show_notification("Sweep timed out.");
		}
	}

	SWP_enable_GUI(true);
}

void GUI_1channel::SWP_draw_plot(QString SWP_raw_data)
{
	QStringList SWP_data = SWP_raw_data.split("\r\n");

	//Remove the OK entry the empty one that comes after.
	QVector<double>	SWP_freq_data;
	QVector<double>	SWP_frw_data;
	QVector<double>	SWP_rfl_data;
	QVector<double>	SWP_s11_dbm_data;
	QVector<double>	SWP_s11_watt_data;

	SWP_freq_data.resize(SWP_data.count()-2);
	SWP_frw_data.resize(SWP_data.count()-2);
	SWP_rfl_data.resize(SWP_data.count()-2);
	SWP_s11_dbm_data.resize(SWP_data.count()-2);
	SWP_s11_watt_data.resize(SWP_data.count()-2);

	QRegExp regexp("\\$SWP\\D?,\\d+,(\\d+.?\\d+),(-?\\d+.?\\d+),(-?\\d+.?\\d+)");


	for (int i = 0; i < SWP_data.count() - 2; i++)
	{
		 qDebug() << SWP_data.at(i) << "#" << i;
		 if (regexp.indexIn(SWP_data.at(i))>=0)
		 {
			  QString string1 = regexp.cap(1);
			  QString string2 = regexp.cap(2);
			  QString string3 = regexp.cap(3);

			  SWP_freq_data[i] = string1.toDouble();
			  SWP_frw_data[i] = string2.toDouble();
			  SWP_rfl_data[i] = string3.toDouble();

			  SWP_s11_dbm_data[i] = SWP_rfl_data[i] - SWP_frw_data[i];
			  SWP_s11_watt_data[i] = (convert_dbm_to_watt(SWP_rfl_data[i]) / convert_dbm_to_watt(SWP_frw_data[i])) * 100;
			  /* Perhaps theoretically a division by zero is possible here, but the minimum power value that can be set is 0.1W / 20dBm anyway...
			   * Since we get our power values in dBm and convert to watt, there's no such thing as 0 dbm anyway... therefor in practice divide by zero doesn't occur. */
		 }
	}

	//Detect best match and visualize it
	SWP_calculate_best_match(SWP_freq_data, SWP_s11_dbm_data);
	ui->label_SWP_best_frequency_2->setText(QString::number(best_S11_match_frequency,'f',2));

	if(!ui->pushButton_SWP_unit->isChecked())	//logarithmic / dBm
	{
		ui->label_SWP_best_match_2->setText(QString::number(best_S11_match_dB,'f',2));
		ui->label_SWP_best_match_3->setText("dB");
	}
	else	//linear / watt
	{
		ui->label_SWP_best_match_2->setText(QString::number(convert_dB_to_percent(best_S11_match_dB),'f',2));
		ui->label_SWP_best_match_3->setText("%");
	}

	//Handle data plotting
	ui->SWP_plot->addGraph();
	ui->SWP_plot->xAxis->setLabel("Frequency (MHz)");
	ui->SWP_plot->xAxis->setRange(SWP_freq_data[0],SWP_freq_data[SWP_freq_data.size()-1]);
	ui->SWP_plot->yAxis->setNumberFormat("f");
	ui->SWP_plot->yAxis->setNumberPrecision(2);
	double min_val, max_val;

	if(!ui->pushButton_SWP_unit->isChecked())	//dbm
	{
		ui->SWP_plot->graph(0)->setData(SWP_freq_data,SWP_s11_dbm_data);

		min_val = *std::min_element(SWP_s11_dbm_data.constBegin(),SWP_s11_dbm_data.constEnd());
		if (min_val > 0){
			 min_val = 0;
		}
		max_val = *std::max_element(SWP_s11_dbm_data.constBegin(),SWP_s11_dbm_data.constEnd());
		if(max_val < 0){
			 max_val = 0;
		}

		ui->SWP_plot->yAxis->setRange(min_val*1.1, max_val*1.1);
		ui->SWP_plot->yAxis->setLabel("S11 (dB)");
		//ui->SWP_plot->yAxis->setRangeReversed(true);
	}
	else
	{
		ui->SWP_plot->graph(0)->setData(SWP_freq_data,SWP_s11_watt_data);

		max_val = *std::max_element(SWP_s11_watt_data.constBegin(),SWP_s11_watt_data.constEnd());

		if(max_val <= 1)
		{
			max_val = 1;
		}
		ui->SWP_plot->yAxis->setRange(0,max_val);
		ui->SWP_plot->yAxis->setLabel("Reflection (%)");
	}

	ui->SWP_plot->setInteractions(QCP::iRangeDrag | QCP::iRangeZoom | QCP::iSelectItems);
	ui->SWP_plot->replot();
}

void GUI_1channel::SWP_calculate_best_match(QVector<double> freq_data, QVector<double> s11_dbm_data)
{
	best_S11_match_frequency = 0;
	best_S11_match_dB = __INT_MAX__;

	for (int i = 0; i < s11_dbm_data.size(); i++)
	{
		if (best_S11_match_dB > s11_dbm_data.at(i))
		{
			best_S11_match_dB = s11_dbm_data.at(i);			//finds the best S11 match (lowest S11 in dB)
			best_S11_match_frequency = freq_data.at(i);	//set frequency to frequency of best S11 match
		}
	}
}


//PSU
void GUI_1channel::handler_PSU_IU_get(int intrasys_num, int psu_num, double voltage, double current, double power)
{
	(void) intrasys_num;

	switch (psu_num)
	{
		case 0:
//			//overview page
//			ui->label_PSU1_voltage_1->setText(QString::number(voltage,'f',1));
//			ui->label_PSU1_current_1->setText(QString::number(current,'f',1));
//			ui->label_PSU1_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU1_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU1_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU1_power_3->setText(QString::number(power,'f',0));
			break;
		case 1:
//			//overview page
//			ui->label_PSU2_voltage_1->setText(QString::number(voltage,'f',1));
//			ui->label_PSU2_current_1->setText(QString::number(current,'f',1));
//			ui->label_PSU2_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU2_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU2_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU2_power_3->setText(QString::number(power,'f',0));
			break;
		case 2:
//			//overview page
//			ui->label_PSU3_voltage_1->setText(QString::number(voltage,'f',1));
//			ui->label_PSU3_current_1->setText(QString::number(current,'f',1));
//			ui->label_PSU3_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU3_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU3_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU3_power_3->setText(QString::number(power,'f',0));
			break;
		case 3:
//			//overview page
//			ui->label_PSU4_voltage_1->setText(QString::number(voltage,'f',1));
//			ui->label_PSU4_current_1->setText(QString::number(current,'f',1));
//			ui->label_PSU4_power_1->setText(QString::number(power,'f',0));
			//PSU page
			ui->label_PSU4_voltage_3->setText(QString::number(voltage,'f',1));
			ui->label_PSU4_current_3->setText(QString::number(current,'f',1));
			ui->label_PSU4_power_3->setText(QString::number(power,'f',0));
			break;
		default:
			break;
	}

	datalogger->setData("PSU_Voltage_" + QString::number(psu_num + 1), QString::number(voltage,'f',1));
	datalogger->setData("PSU_Current_" + QString::number(psu_num + 1), QString::number(current,'f',1));
	datalogger->setData("PSU_Power_" + QString::number(psu_num + 1), QString::number(power,'f',1));
}

void GUI_1channel::handler_PSU_power_efficiency_get(int intrasys_num, double val)
{
	(void) intrasys_num;

//	ui->label_PSU_efficiency_2->setText(QString::number(val, 'f',1));
	ui->label_PSU_efficiency_5->setText(QString::number(val, 'f',1));
	datalogger->setData("PSU_Power_Efficiency", QString::number(val, 'f',1));
}

void GUI_1channel::handler_PSU_voltage_setpoint_get(int intrasys_num, double val)
{
	(void) intrasys_num;

	if (val >= 0)
	{
		ui->pushButton_PSU_voltage_1->setText(zeroChopper(QString::number(val,'f',1)));
	}
	else
	{
		ui->pushButton_PSU_voltage_1->setText("-");
	}
}

void GUI_1channel::handler_PSU_current_limit_get(int intrasys_num, double val)
{
	(void) intrasys_num;

	if (val >= 0)
	{
		ui->pushButton_PSU_current_limit_1->setText(zeroChopper(QString::number(val,'f',1)));
	}
	else
	{
		ui->pushButton_PSU_current_limit_1->setText("-");
	}
}

void GUI_1channel::handler_PSU_enable_combined_get(int intrasys_num, bool enable)
{
	(void) intrasys_num;

	//
	//TODO:
	//PSU enable state change is not very elegant.
	//All PSU's are shown as enabled; even though there may be less present.
	//Data logging trigger is kind of... indirect?
	//

	QString enable_string = "";
	enable == true ? enable_string = "ON" : enable_string = "OFF";

	//Enable/Disable datalogging when PSU enable state changes;
//	if (ui->pushButton_PSU_enable_1->text() != enable_string)
//	{
//		datalogger->enable_datalogging(enable);
//	}

//	//overview page
//	ui->label_PSU1_enable_1->setText(enable_string);
//	ui->label_PSU2_enable_1->setText(enable_string);
//	ui->label_PSU3_enable_1->setText(enable_string);
//	ui->label_PSU4_enable_1->setText(enable_string);

	//PSU page
	ui->pushButton_PSU_enable_1->setText(enable_string);
	ui->pushButton_PSU_enable_1->setChecked(enable);
	ui->label_PSU1_enable_2->setText(enable_string);
	ui->label_PSU2_enable_2->setText(enable_string);
	ui->label_PSU3_enable_2->setText(enable_string);
	ui->label_PSU4_enable_2->setText(enable_string);

//#if defined (Q_OS_LINUX)
//	/* Toggle interlock label visibility when there is a change. */
//	ui->label_interlock_1->setEnabled(!enable);
//	ui->label_interlock_1->setVisible(!enable);
//#endif
}

void GUI_1channel::handler_PSU_dissipation_get(int intrasys_num, double val)
{
	(void) intrasys_num;

//	ui->label_PSU_dissipation_2->setText(zeroChopper(QString::number(val,'f',1)));
	ui->label_PSU_dissipation_5->setText(zeroChopper(QString::number(val,'f',1)));
	datalogger->setData("PSU_Power_Dissipation", QString::number(val, 'f',1));
}

void GUI_1channel::handler_PSU_voltage_measurement_get(int intrasys_num, double PSU_voltage_combined)
{
	(void) intrasys_num;

	datalogger->setData(13, QString::number(PSU_voltage_combined));
}

void GUI_1channel::handler_PSU_current_measurement_get(int intrasys_num, double PSU_current_combined)
{
	(void) intrasys_num;

	datalogger->setData(14, QString::number(PSU_current_combined));
}

/* Reconfigure Power Supplies */
void GUI_1channel::PSU_reset()
{
	//
	// TODO:
	// These configuration actions violate the design ethos of the 2.X GUI.
	// They exist as a workaround for ZHL I2C issues, and hardware boot-up timing issues.
	// But they can result in conflicts against the EEPROM content of the SGx device.
	// When a better solution is available, these functionality should be removed.
	//

	if (config->get_PSU_count() > 0)
	{
		if (config->get_PSU_voltage() > 0)
		{
			emit signal_setPSU_voltage(1, config->get_PSU_voltage());
		}
		if (config->get_PSU_current_limit() > 0)
		{
			emit signal_setPSU_current_limit(1, config->get_PSU_current_limit());
		}
		if (config->get_PSU_enabled() >= 0)
		{
			emit signal_setPSU_enable(1, (bool)config->get_PSU_enabled());
		}
	}
}

/**********************************************************************************************************************************************************************************
 * Show mode settings
 * *******************************************************************************************************************************************************************************/
/* Receive a 'vote' for the feature availability and visibility from RF Channel */
void GUI_1channel::handler_availability_amplifier_mode(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	availability_amplifier_mode = available;
	visibility_amplifier_mode = visible;
	configure_availability_amplifier_mode();
}

/* Combines 'votes' for availability and visibility of the feature */
void GUI_1channel::configure_availability_amplifier_mode()
{
	bool available = availability_amplifier_mode;
	bool visible = visibility_amplifier_mode;

	if (remote_command_mode == RCM_MODE::RCM_MODE_USB_BLIND ||
		remote_command_mode == RCM_MODE::RCM_MODE_TCP_BLIND ||
		remote_command_mode == RCM_MODE::RCM_MODE_RS232_BLIND ||
		remote_command_mode == RCM_MODE::RCM_MODE_RS485_BLIND
		)
	{
		available &= false;
	}
	else
	{
		available &= true;
	}

	// Set Visibility and Enable state
	ui->pushButton_power_control_amplifier_1->setVisible(visible);
	ui->pushButton_power_control_amplifier_1->setEnabled(available);
}

/* Receive a 'vote' for the feature availability and visibility from RF Channel */
void GUI_1channel::handler_availability_external_triggering(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	availability_ext_triggering = available;
	visibility_ext_triggering = visible;
	configure_availability_external_triggering();
}

/* Combines 'votes' for availability and visibility of the feature */
void GUI_1channel::configure_availability_external_triggering()
{
	bool available = availability_ext_triggering;
	bool visible = visibility_ext_triggering;

	if (remote_command_mode == RCM_MODE::RCM_MODE_USB_BLIND ||
		remote_command_mode == RCM_MODE::RCM_MODE_TCP_BLIND ||
		remote_command_mode == RCM_MODE::RCM_MODE_RS232_BLIND ||
		remote_command_mode == RCM_MODE::RCM_MODE_RS485_BLIND
		)
	{
		available &= false;
	}
	else
	{
		available &= true;
	}

	// Set Visibility and Enable state
	ui->External_triggering_label->setVisible(visible);
	ui->pushButton_external_triggering_ON_1->setVisible(visible);
	ui->pushButton_external_triggering_OFF_1->setVisible(visible);

	ui->External_triggering_label->setEnabled(available);
	ui->pushButton_external_triggering_ON_1->setEnabled(available);
	ui->pushButton_external_triggering_OFF_1->setEnabled(available);
}

/* Availability handler for the PWM enable button. */
void GUI_1channel::handler_availability_PWM(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	ui->pushButton_PWM_enable_1->setVisible(visible);
	ui->pushButton_PWM_enable_1->setEnabled(available);
}

/* Availability handler for the ALL enable button. */
void GUI_1channel::handler_availability_ALL(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	ui->pushButton_ALL_enable_1->setVisible(visible);
	ui->pushButton_ALL_enable_1->setEnabled(available);
}

/* Availability handler for the DLL enable button. */
void GUI_1channel::handler_availability_DLL(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	ui->pushButton_DLL_enable_1->setVisible(visible);
	ui->pushButton_DLL_enable_1->setEnabled(available);
}

void GUI_1channel::handler_availability_sweep(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	ui->pushButton_SWP_execute_1->setEnabled(available);
	ui->pushButton_SWP_execute_2->setEnabled(available);

	/* Important: The visibility of the SWP_execute_1 button also depends on some purely GUI-related factors.
	 * Namely, it is only made visible whenever the user is located in the Sweep Menu.
	 * That part can only be managed by the GUI, so the 'visible' value is stored and reused elsewhere */

	sweep_visible = visible;
	show_sweep_button();
}

void GUI_1channel::handler_availability_PSU_enable(int intrasys_num, bool visible, bool available)
{
	(void) intrasys_num;

	ui->pushButton_PSU_enable_1->setVisible(visible);
	ui->pushButton_PSU_enable_1->setEnabled(available);
}

void GUI_1channel::show_PWM_settings(bool input)
{
	ui->PWM_freq_label->setVisible(input);
	ui->PWM_dc_label->setVisible(input);
	ui->pushButton_PWM_frequency_1->setVisible(input);
	ui->pushButton_PWM_duty_cycle_1->setVisible(input);
}

void GUI_1channel::show_DLL_settings(bool input)
{
	ui->DLL_stepFreq_label->setVisible(input);
	ui->DLL_threshold_label->setVisible(input);
	ui->pushButton_DLL_step_frequency_1->setVisible(input);
	ui->pushButton_DLL_threshold_1->setVisible(input);
	ui->label_DLL_frequency_lock_1->setVisible(input);
	ui->label_DLL_frequency_lock_2->setVisible(input);
	ui->label_DLL_frequency_lock_3->setVisible(input);
}

void GUI_1channel::show_ALL_settings(bool input)
{
	ui->ALL_threshold_label->setVisible(input);
	if (config->get_support_power_offset())
	{
		ui->label_3->setVisible(!input);
	}
	else
	{
		ui->label_3->setVisible(input);
	}
	ui->pushButton_ALL_threshold_1->setVisible(input);
}

void GUI_1channel::show_autogain_controls(int intrasys_num, bool enable, bool visible)
{
	(void) intrasys_num;

	/* enable/disable power controls */
	ui->pushButton_power_dbm_1->setEnabled(enable);
	ui->pwr_dbm_label->setEnabled(enable);
	ui->pushButton_power_watt_1->setEnabled(enable);
	ui->pwr_watt_label->setEnabled(enable);

	ui->pushButton_power_dbm_2->setEnabled(enable);
	ui->DLL_power_dbm_label->setEnabled(enable);
	ui->pushButton_power_watt_2->setEnabled(enable);
	ui->DLL_power_watt_label->setEnabled(enable);

	ui->pushButton_power_dbm_1->setVisible(visible);
	ui->pwr_dbm_label->setVisible(visible);
	ui->pushButton_power_watt_1->setVisible(visible);
	ui->pwr_watt_label->setVisible(visible);

	//DO NOT ADD FEED FORWARD CONTROLS HERE, GIVE IT A SEPARATE FUNCTION SO THAT YOU CAN KEEP CONTROLS BLOCK WHILE PID / ANALOGING
	//Ideally separate functions for GCS and MCS and PWRSGD... analog control with IQ Mod, could benefit from GCS controls only for example...
}

void GUI_1channel::show_sweep_power_controls(int intrasys_num, bool enable, bool visible)
{
	(void) intrasys_num;

	ui->SWP_power_dbm_label->setEnabled(enable);
	ui->SWP_power_watt_label->setEnabled(enable);
	ui->pushButton_SWP_power_dbm_1->setEnabled(enable);
	ui->pushButton_SWP_power_watt_1->setEnabled(enable);

	ui->SWP_power_dbm_label->setVisible(visible);
	ui->SWP_power_watt_label->setVisible(visible);
	ui->pushButton_SWP_power_dbm_1->setVisible(visible);
	ui->pushButton_SWP_power_watt_1->setVisible(visible);
}

void GUI_1channel::show_sweep_button()
{
	bool factor_sweep_page = false;
	bool factor_availability = sweep_visible;

	if (ui->stackedWidget->currentIndex() == ui->stackedWidget->indexOf(ui->sweep_controls_page)
		|| ui->stackedWidget_3->currentIndex() == ui->stackedWidget_3->indexOf(ui->Sweep_graph_page))
	{
		factor_sweep_page = true;
	}

	ui->pushButton_SWP_execute_1->setVisible(factor_sweep_page & factor_availability);
	ui->pinkline_2->setVisible(factor_sweep_page & factor_availability);
}

/* Disable any controls that utilize the IQ Modulator. Also hide the RF Enable button. It is not indicative of anything in this mode. */
void GUI_1channel::show_amplifier_mode_controls(int intrasys_num, bool enable)
{
	(void) intrasys_num;

	/* Remark: Frequency button is left available so that it can be manually matched to the frequency of the external oscillator source for more accurate calibration / power measurements. */

	switch(config->get_support_amplifier_mode())
	{
		case 1:		//Native Amplifier Mode - SG hardware supports Amplifier mode natively.

			//Nothing special to do...
			break;

		case 2:		//Bypass Amplifier Mode - External signal bypasses the controller and goes straight to the PA.

			ui->pushButton_RF_enable_1->setEnabled(!enable);
			ui->pushButton_RF_enable_1->setVisible(!enable);
			break;

		case 3:		//SSG Amplifier Mode - Requires modified SGx board.

			ui->pushButton_RF_enable_1->setEnabled(!enable);
			ui->pushButton_RF_enable_1->setVisible(!enable);
			break;

		default:	
			break;
	}

	/* Toggle the enable state of the phase control
	 * The phase control should be hidden during amplifier mode
	 * UNLESS phase control is explicitly permitted according to the config item support_AM_phase_control */
	ui->pushButton_phase_1->setEnabled((config->get_support_AM_phase_control() || !enable) && config->get_support_phase_control());
	ui->phase_label->setEnabled((config->get_support_AM_phase_control() || !enable) && config->get_support_phase_control());
}

void GUI_1channel::show_GCS_controls(int intrasys_num, bool enable, bool visible)
{
	(void) intrasys_num;

	ui->label_VGA_attenuation_1->setEnabled(enable);
	ui->pushButton_VGA_attenuation_1->setEnabled(enable);
	ui->label_VGA_attenuation_1->setVisible(visible);
	ui->pushButton_VGA_attenuation_1->setVisible(visible);
}

void GUI_1channel::show_MCS_controls(int intrasys_num, bool enable, bool visible)
{
	(void) intrasys_num;

	ui->label_IQMod_magnitude_1->setEnabled(enable);
	ui->pushButton_IQMod_magnitude_1->setEnabled(enable);
	ui->label_IQMod_magnitude_1->setVisible(visible);
	ui->pushButton_IQMod_magnitude_1->setVisible(visible);
}

/**********************************************************************************************************************************************************************************
 * VERSION COMPATIBILITY CHECK
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::firmware_version_check(QString firmware_version)
{
	QStringList ver = firmware_version.split(".");
	std::array<int,4> firmware_version_actual;
	firmware_version_actual[0] = ver.at(0).toInt();
	firmware_version_actual[1] = ver.at(1).toInt();
	firmware_version_actual[2] = ver.at(2).toInt();

	if (ver.count() > 3)
	{
		firmware_version_actual[3] = ver.at(3).toInt();
	}
	else
	{
		firmware_version_actual[3] = 0;
	}

	if (isVersionCompatible(firmware_version_actual, firmware_version_requirement) == false)
	{
		QString comp_warning =	"The current firmware version is unsupported by this application and may result in unpredictable behaviour."
								"Please upgrade the firmware of your SGx board to at least the recommended version.\n"
								"\nRecommended firmware: v";
//		QString comp_warning =	"The feature-set of this application requires a different firmware version.\n"
//								"Please upgrade the firmware of your SGx board to at least the recommended version.\n"
//								"\nRecommended firmware: v";

		for (unsigned int i = 0; i < firmware_version_requirement.size(); i++)
		{
			comp_warning += QString::number(firmware_version_requirement.at(i)) + ".";
		}
		comp_warning.chop(1);	//chop the redundant period
		if (firmware_version_requirement.at(3) == 0)
		{
			comp_warning.chop(2);	//If the final hotfix number the required version is 0, don't show it.
		}
		comp_warning += "\nCurrent firmware: " + firmware_version +
						"\n\nProceed at own risk.";

		QMessageBox message;
		message.warning(nullptr,	"Software imcompatibility", comp_warning);
		show_notification("Firmware is outdated!\nProceed at own risk.");
	}
}

/**********************************************************************************************************************************************************************************
 * NOTIFICATION / WARNING BAR
 * *******************************************************************************************************************************************************************************/
void GUI_1channel::handler_notification_get(int intrasys_num, QString notification)
{
	(void) intrasys_num;

	show_notification(notification);
}

void GUI_1channel::show_notification(QString message, int time)
{
	notification_timer->setSingleShot(true);
	connect(notification_timer, SIGNAL(timeout()), this, SLOT(close_notification()));
	ui->label_notification->setText(message);
	ui->label_notification->setVisible(true);
	notification_timer->start(time);
}

void GUI_1channel::close_notification()
{
	ui->label_notification->setVisible(false);
}

void GUI_1channel::show_error(QString message)
{
	ui->label_error_message->setText(message);
	ui->label_error_message->setVisible(true);
}

void GUI_1channel::close_error()
{
	ui->label_error_message->setVisible(false);
}


/**********************************************************************************************************************************************************************************
 * Logging
 * *******************************************************************************************************************************************************************************/

//
// TODO:
// prevent duplicate inputs; requires some kind of state check of the datalogger...
//
void GUI_1channel::on_pushButton_data_logging_OFF_1_clicked()
{
	datalogger->enable_datalogging(false);
}

void GUI_1channel::on_pushButton_data_logging_ON_1_clicked()
{
	datalogger->enable_datalogging(true);
}

void GUI_1channel::handler_datalogging_enable_get(bool enable)
{
	if (enable == true)
	{
		ui->pushButton_data_logging_ON_1->setChecked(true);
		ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Enabled");
	}
	else
	{
		ui->pushButton_data_logging_OFF_1->setChecked(true);
		ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Disabled");
	}
}

/* Set the start condition for Logging */
void GUI_1channel::handler_datalogging(bool enable)
{
	if (enable)
	{
		ui->pushButton_data_logging_ON_1->click();
	}
	else
	{
		ui->pushButton_data_logging_OFF_1->click();
	}
}

void GUI_1channel::handler_datalogging_storage_sufficient(bool valid)
{
	ui->label_low_storage_1->setVisible(!valid);
}

void GUI_1channel::handler_datalogging_log_deleted(bool success, QString filename)
{
	ui->settings_plainTextEdit->appendPlainText(">\tInsufficient storage for logging. Deleting older log.");
	if (success)
	{
		ui->settings_plainTextEdit->appendPlainText(">\tDeleted old log: " + filename);
	}
	else
	{
		ui->settings_plainTextEdit->appendPlainText(">\tCould not delete file: " + filename);
	}
}

//void GUI_1channel::on_pushButton_data_logging_ON_1_toggled(bool checked)
//{
//	if (datalogger->enable_datalogging(checked) == true)
//	{
//		if (checked)
//			ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Enabled");
//		else
//			ui->settings_plainTextEdit->appendPlainText(">\tDatalogging Disabled");
//	}
//	else
//	{
//		ui->settings_plainTextEdit->appendPlainText(">\tError: unable to start Datalogging");
//	}
//}

/**********************************************************************************************************************************************************************************
 * HELP
 * *******************************************************************************************************************************************************************************/
/* Manage the contents of the textBrowser in the Help menu */
void GUI_1channel::on_listWidget_currentItemChanged(QListWidgetItem *current, QListWidgetItem *previous)
{
	(void) previous;

	QString path_to_file;
	QFile file;

	if (current == &listWidgetItem_system)
	{
		update_listWidgetItem_system();
		return;		//Exit function; There is no Qfile to open in this case;
	}
	else if (current == &listWidgetItem_license)
	{
		path_to_file = QCoreApplication::applicationDirPath() + "/legal/GNU_GPLv3_License.html";
	}
	else if (current == &listWidgetItem_about)
	{
		path_to_file = QCoreApplication::applicationDirPath() + "/legal/notice.html";
	}
	else
	{
		path_to_file = "";
	}

	file.setFileName(path_to_file);

	/* Check if the requested file can be found, otherwise fall back to the compiled-in backup. */
	if (!file.exists())
	{
		path_to_file.replace(QCoreApplication::applicationDirPath(), ":");
		file.setFileName(path_to_file);
	}

	if (file.exists())
	{
		if (file.open(QFile::ReadOnly | QFile::Text))
		{
			QTextStream stream(&file);
			ui->textBrowser->setHtml(file.readAll());
		}
	}
	else
	{
		ui->textBrowser->clear();
	}
}

/* Update the content for the listWidgetItem 'System' in the help menu. */
void GUI_1channel::update_listWidgetItem_system()
{
	system_string = QString("<body style=\" font-family:'Arial';\">"
					"<b>Toolkit GUI version:</b><br>"	+ toolkit_GUI_version	+ "<br>" + "<br>" +
					"<b>SSG firmware version:</b><br>"	+ SG_firmware_version	+ "<br>" + "<br>" +
					"<b>SSG manufacturer:</b><br>"		+ SG_manufacturer		+ "<br>" + "<br>" +
					"<b>SSG model:</b><br>"				+ SG_board_model		+ "<br>" + "<br>" +
					"<b>SSG serial number:</b><br>"		+ SG_serial_number		+ "<br>" + "<br>");

	if (config->get_report_PA_type())
	{
		system_string += QString ("<b>PA type:</b><br>"	+ QString::number(PA_type));
	}

	system_string += QString ("</body>");

	if (ui->listWidget->currentItem() == &listWidgetItem_system)
	{
		ui->textBrowser->setHtml(system_string);
	}
}
